package de.duehl.vocabulary.japanese.tools;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.jupiter.api.Test;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.datetime.date.ImmutualDate;
import de.duehl.vocabulary.japanese.VocabularyTrainerVersion;
import de.duehl.vocabulary.japanese.common.data.InternalAdditionalVocableData;
import de.duehl.vocabulary.japanese.common.data.VocablesShuffleType;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.FumikoDataStructures;
import de.duehl.vocabulary.japanese.data.Vocable;

import static org.junit.jupiter.api.Assertions.*;
import static de.duehl.vocabulary.japanese.tools.VocablesForTesting.*;

public class VocableListShufflerTestGermanToJapanese {

    private static final InternalAdditionalVocableData MAKURA_DATA =
            new InternalAdditionalVocableData()
                    .setKey("まくらKopfkissen")
                    .setGermanToJapaneseTestCount(57)
                    .setCorrectGermanToJapaneseTestCount(54)
                    .setFirstSeenDate(new ImmutualDate("11.10.2023"))
                    .setLastGermanToJapaneseTestDate(new ImmutualDate("21.10.2023"))
                    .setLastCorrectGermanToJapaneseTestDate(new ImmutualDate("21.10.2023"))
                    .initLastTenGermanToJapaneseTestResultsFromStorageString("++++++++++")
                    ;

    @Test
    public void sortAlphabetical() {
        List<Vocable> vocables = CollectionsHelper.buildListFrom(MAKURA, KAGAMI, AI, SEKKEN, BEDDO,
                FUTON, DEMO, KAGE);

        Map<Vocable, InternalAdditionalVocableData> requesterMap = new HashMap<>();
        for (Vocable vocable : vocables) {
            requesterMap.put(vocable, MAKURA_DATA);
        }

        Options options = new Options(new VocabularyTrainerVersion().getVersion());
        options.setReverseShuffledVocables(false);

        FumikoDataStructures dataStructures = TestDataStructureBuilder.create(requesterMap,
                options);

        VocableListShuffler shuffler = new VocableListShuffler(vocables,
                VocablesShuffleType.ALPHABETICAL, dataStructures);
        shuffler.shuffle();
        List<Vocable> shuffled = shuffler.getVocables();
        assertEquals(8, shuffled.size());
        assertEquals(AI, shuffled.get(0));
        assertEquals(BEDDO, shuffled.get(1));
        assertEquals(DEMO, shuffled.get(2));
        assertEquals(FUTON, shuffled.get(3));
        assertEquals(KAGAMI, shuffled.get(4));
        assertEquals(KAGE, shuffled.get(5));
        assertEquals(MAKURA, shuffled.get(6));
        assertEquals(SEKKEN, shuffled.get(7));
    }

    @Test
    public void sortReverseAlphabetical() {
        List<Vocable> vocables = CollectionsHelper.buildListFrom(MAKURA, KAGAMI, AI, SEKKEN, BEDDO,
                FUTON, DEMO, KAGE);

        Map<Vocable, InternalAdditionalVocableData> requesterMap = new HashMap<>();
        for (Vocable vocable : vocables) {
            requesterMap.put(vocable, MAKURA_DATA);
        }

        Options options = new Options(new VocabularyTrainerVersion().getVersion());
        options.setReverseShuffledVocables(true);

        FumikoDataStructures dataStructures = TestDataStructureBuilder.create(requesterMap,
                options);

        VocableListShuffler shuffler = new VocableListShuffler(vocables,
                VocablesShuffleType.ALPHABETICAL, dataStructures);
        shuffler.shuffle();
        List<Vocable> shuffled = shuffler.getVocables();
        assertEquals(8, shuffled.size());
        assertEquals(SEKKEN, shuffled.get(0));
        assertEquals(MAKURA, shuffled.get(1));
        assertEquals(KAGE, shuffled.get(2));
        assertEquals(KAGAMI, shuffled.get(3));
        assertEquals(FUTON, shuffled.get(4));
        assertEquals(DEMO, shuffled.get(5));
        assertEquals(BEDDO, shuffled.get(6));
        assertEquals(AI, shuffled.get(7));
    }

}
