package de.duehl.basics.datetime.time.watch;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.io.Serializable;

import de.duehl.basics.datetime.time.TimeHelper;

/**
 * Eine Klasse zur Zeitmessung eines Countdowns.
 *
 * @version 1.01     2021-03-17
 * @author Christian Dühl
 */

public class CountDownWatch implements WatchBase, Serializable {

    private static final long serialVersionUID = 1L;

    /** Startzeit in Millisekunden. */
    private long startTime;

    /** Laufzeit des Countdowns ins Millisekunden. */
    private long duration;

    /**
     * Konstruktor.
     *
     * Startet den Countdown automatisch.
     *
     * @param duration
     *            Dauer des Countdowns in der Form HH:MM:SS.
     */
    public CountDownWatch(String duration) {
        this(TimeHelper.hoursMinutesSecondsToSeconds(duration));
    }

    /**
     * Konstruktor.
     *
     * Startet den Countdown automatisch.
     *
     * @param duration
     *            Dauer des Countdowns in Sekunden.
     */
    public CountDownWatch(long duration) {
        setDurationInSeconds(duration);
        start();
    }

    /** Startet den Countdown. */
    public void start() {
        startTime = TimeHelper.getCurrentTimeInMilliseconds();
    }

    /** Setter für die Laufzeit des Countdowns ins Millisekunden. */
    public void setDurationInMilliseconds(long durationMillis) {
        this.duration = durationMillis;
    }

    /** Setter für die Laufzeit des Countdowns ins Millisekunden. */
    public void setDurationInSeconds(long durationSeconds) {
        setDurationInMilliseconds(durationSeconds * 1000);
    }

    /**
     * Liest die Zeit auf dem Countdown und gibt sie in einem von Menschen lesbarem Format zurück.
     *
     * @return Zeit des Countdowns im Format HH:MM:SS.
     */
    @Override
    public String getTime() {
        return TimeHelper.secondsToHoursMinutesSeconds(getTimeInSeconds());
    }

    /** Gibt die Zeit auf dem Countdown in Sekunden zurück. */
    @Override
    public long getTimeInSeconds() {
        return getTimeInMilliSeconds() / 1000;
    }

    /** Gibt die Zeit auf dem Countdown in Millisekunden zurück. */
    long getTimeInMilliSeconds() {
        long now = TimeHelper.getCurrentTimeInMilliseconds();
        long durationSinceStart = now - startTime;
        if (durationSinceStart > duration) {
            return 0;
        }
        else {
            return duration - durationSinceStart;
        }
    }

    /** Stringrepräsentation. */
    @Override
    public String toString() {
        return "CountDownWatch [" + getTime() + "]";
    }

}
