package de.duehl.basics.logging;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.datetime.Timestamp;
import de.duehl.basics.io.FileHelper;
import de.duehl.basics.logging.eraser.LogEraser;
import de.duehl.basics.text.Text;

/**
 * Dieses Klasse dient zum direkten Ausgeben von Nachrichten in ein Log ohne Angabe der Methoden
 * und Zeilen. Normalerweise möchte man dies nicht, sondern den FileLogger verwenden!
 *
 * @version 1.01     2022-04-27
 * @author Christian Dühl
 */

public class PlainMessageToFileLogger implements Logger {

    /** Name des Logfiles. */
    private final String logFileName;

    public PlainMessageToFileLogger(String logFileName) {
        this.logFileName = logFileName;
        FileHelper.createEmptyFile(logFileName); // wir Überschrieben im Zweifelsfall auch!
    }

    /** Loggt die übergebene Nachricht. Diese darf keinen Zeilenumbruch enthalten! */
    @Override
    public void log(String text) {
        String textToLog = text;
        textToLog = Text.removeLineBreaks(textToLog);
        textToLog = Text.tabToSpace(textToLog);
        FileLogger.checkNoLineBreaks(textToLog);
        FileHelper.appendLineToFile(textToLog, logFileName);
    }

    @Override
    public void log(String text, int stacktraceOffset) {
        throw new RuntimeException("Der PlainMessageToFileLogger verwendet keinen Stacktrace!");
    }

    /** Getter für den Namen des Logfiles. */
    @Override
    public String getLogFileName() {
        return logFileName;
    }

    /** Gibt den Pfad zurück, in dem die Logdatei liegt. */
    @Override
    public String getLogPath() {
        return FileHelper.getDirName(logFileName);
    }

    @Override
    public int getLastWrittenLineNumber() {
        throw new RuntimeException("Der PlainMessageToFileLogger merkt sich keine Zeilennummern!");
    }

    /**
     * Erzeugt einen Logger mit einem leeren Logfile und entfernt gleich passend alte Logdateien.
     *
     * @param begin
     *            Anfang des Logdateinamen.
     * @param end
     *            Ende des Logdateinamen.
     * @param path
     *            Pfad zur Logdatei.
     * @return Logger-Objekt.
     * @throws IllegalArgumentException
     *             Falls der Pfad kein bestehendes Verzeichnis ist.
     */
    public static Logger create(String begin, String end, String path) {
        FileLogger.checkIfPathExistsAndIsDirectory(path);

        String pureFilename = begin + Timestamp.fullTimestamp() + end;
        String fileName = FileHelper.concatPathes(path, pureFilename);

        Logger logger = new PlainMessageToFileLogger(fileName);

        LogEraser eraser = new LogEraser(path, begin, end, logger);
        eraser.erase();

        return logger;
    }

}
