package de.duehl.basics.system;

/*
 * Copyright 2016 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.io.PrintStream;

/**
 * Diese Klasse kann System.out und System.err umleiten.
 *
 * @version 1.01     2016-01-21
 * @author Christian Dühl
 */

public class OutputStreamSaveAndRelease {

    private static OutputStream NOT_SAVED_STREAM = new ByteArrayOutputStream();

    /** Originaler Stream für System.out. */
    private OutputStream originalOut;

    /** Originaler Stream für System.err. */
    private OutputStream originalErr;

    /** Neuer Stream für System.out. */
    private OutputStream out;

    /** Neuer Stream für System.err. */
    private OutputStream err;

    /** Konstruktor. */
    public OutputStreamSaveAndRelease() {
        originalOut = NOT_SAVED_STREAM;
        originalErr = NOT_SAVED_STREAM;
        out = NOT_SAVED_STREAM;
        err = NOT_SAVED_STREAM;
    }

    /** Sichert den originalen Strom System.out und lenkt System.out auf einen neuen um. */
    public void saveSystemOut() {
        if (originalOut != NOT_SAVED_STREAM) {
            throw new RuntimeException("Zwischen zwei saveSystemOut()-Aufrufen muss ein "
                    + "releaseSystemOut()-Aufruf erfolgen!");
        }
        originalOut = System.out;

        out = new ByteArrayOutputStream();
        System.setOut(new PrintStream(out));
    }

    /** Setzt System.out wieder auf den Standard. */
    public void releaseSystemOut() {
        if (originalOut == NOT_SAVED_STREAM) {
            throw new RuntimeException("Vor einem releaseSystemOut()-Aufruf muss ein "
                    + "saveSystemOut()-Aufruf erfolgen!");
        }
        System.setOut(new PrintStream(originalOut));
        originalOut = NOT_SAVED_STREAM;
    }

    /** Liefert den während dem Test nach System.out geschriebenen Text. */
    public String getOutContents() {
        if (originalOut == NOT_SAVED_STREAM) {
            throw new RuntimeException("Vor einem releaseSystemOut()-Aufruf muss ein "
                    + "saveSystemOut()-Aufruf erfolgen!");
        }
        return out.toString();
    }

    /** Sichert den originalen Strom System.err und lenkt System.err auf einen neuen um. */
    public void saveSystemErr() {
        if (originalErr != NOT_SAVED_STREAM) {
            throw new RuntimeException("Zwischen zwei saveSystemErr()-Aufrufen muss ein "
                    + "releaseSystemErr()-Aufruf erfolgen!");
        }
        originalErr = System.err;

        err = new ByteArrayOutputStream();
        System.setErr(new PrintStream(err));
    }

    /** Setzt System.err wieder auf den Standard. */
    public void releaseSystemErr() {
        if (originalErr == NOT_SAVED_STREAM) {
            throw new RuntimeException("Vor einem releaseSystemErr()-Aufruf muss ein "
                    + "saveSystemErr()-Aufruf erfolgen!");
        }
        System.setErr(new PrintStream(originalErr));
        originalErr = NOT_SAVED_STREAM;
    }

    /** Liefert den während dem Test nach System.err geschriebenen Text. */
    public String getErrContents() {
        if (originalErr == NOT_SAVED_STREAM) {
            throw new RuntimeException("Vor einem getErrContents()-Aufruf muss ein "
                    + "saveSystemErr()-Aufruf erfolgen!");
        }
        return err.toString();
    }

    @Override
    public String toString() {
        return "OutputStreamSaveAndRelease ["
                + "originalOut=" + originalOut
                + ", originalErr=" + originalErr
                + ", out=" + out
                + ", err=" + err
                + ", NOT_SAVED_STREAM=" + NOT_SAVED_STREAM + "]";
    }

    /** Gibt eine Nachricht auf dem originalen System.out aus, wenn dies gesichert ist. */
    public void printOut(String message) {
        if (originalOut == NOT_SAVED_STREAM) {
            throw new RuntimeException("Die Methode printOut() kann nur benutzt werden, nachdem "
                    + "ein saveSystemOut()-Aufruf erfolgt ist!");
        }
        new PrintStream(originalOut).println(message);
    }

}
