package de.duehl.basics.text.data;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.io.FileHelper;
import de.duehl.basics.text.NumberString;
import de.duehl.basics.text.Text;

/**
 * Diese Klasse stellt die Information über den freien und gesamten Speicherplatz an einem Ort dar.
 *
 * @version 1.01     2020-03-03
 * @author Christian Dühl
 */

public class FreeSpaceInfo {

    private final long totalDiscSpace;
    private final long freeDiscSpace;

    private final long totalDiscSpaceKb;
    private final long freeDiscSpaceKb;

    private final long totalDiscSpaceMb;
    private final long freeDiscSpaceMb;

    private final long totalDiscSpaceGb;
    private final long freeDiscSpaceGb;

    private final double percentFree;
    private final String percentFreeDescription;

    public FreeSpaceInfo(String path) {
        totalDiscSpace = FileHelper.totalDiscSpace(path);
        freeDiscSpace = FileHelper.freeDiscSpace(path);

        totalDiscSpaceKb = totalDiscSpace / 1024L;
        freeDiscSpaceKb = freeDiscSpace / 1024L;

        totalDiscSpaceMb = totalDiscSpaceKb / 1024L;
        freeDiscSpaceMb = freeDiscSpaceKb / 1024L;

        totalDiscSpaceGb = totalDiscSpaceMb / 1024L;
        freeDiscSpaceGb = freeDiscSpaceMb / 1024L;

        percentFree = NumberString.percentAsNumber(freeDiscSpace, totalDiscSpace);
        percentFreeDescription = NumberString.twoDecimalPlaces(percentFree);
    }

    public long getTotalDiscSpace() {
        return totalDiscSpace;
    }


    public long getFreeDiscSpace() {
        return freeDiscSpace;
    }


    public long getTotalDiscSpaceKb() {
        return totalDiscSpaceKb;
    }


    public long getFreeDiscSpaceKb() {
        return freeDiscSpaceKb;
    }


    public long getTotalDiscSpaceMb() {
        return totalDiscSpaceMb;
    }


    public long getFreeDiscSpaceMb() {
        return freeDiscSpaceMb;
    }


    public long getTotalDiscSpaceGb() {
        return totalDiscSpaceGb;
    }


    public long getFreeDiscSpaceGb() {
        return freeDiscSpaceGb;
    }


    public double getPercentFree() {
        return percentFree;
    }


    public String getPercentFreeDescription() {
        return percentFreeDescription;
    }


    public String totalMb() {
        return Text.fillWithSpacesAtFront(NumberString.taupu(totalDiscSpaceMb), 10) + " MB";
    }

    public String freeMb() {
        return Text.fillWithSpacesAtFront(NumberString.taupu(freeDiscSpaceMb), 10)+ " MB"
                + " (" + percentFreeDescription + "%)";
    }

}
