package de.duehl.basics.datetime;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import org.junit.Test;

import de.duehl.basics.datetime.date.ImmutualDate;
import de.duehl.basics.datetime.time.ImmutualTime;

import static de.duehl.basics.datetime.DateAndTime.SECONDS_OF_ONE_DAY;

public class DateAndTimeTest {

    @Test
    public void create() {
        DateAndTime dateAndTime = new DateAndTime();
        assertNotNull(dateAndTime);
    }

    @Test
    public void createWithValues() {
        ImmutualDate date = new ImmutualDate(30, 12, 2016);
        ImmutualTime time = new ImmutualTime( 9, 46, 23);
        DateAndTime dateAndTime = new DateAndTime(date, time);
        assertNotNull(dateAndTime);
    }

    @Test
    public void getTime() {
        DateAndTime dateAndTime = new DateAndTime();
        ImmutualTime time = dateAndTime.getTime();
        assertNotNull(time);
    }

    @Test
    public void getDate() {
        DateAndTime dateAndTime = new DateAndTime();
        ImmutualDate date = dateAndTime.getDate();
        assertNotNull(date);
    }

    @Test
    public void getTime2() {
        ImmutualDate date = new ImmutualDate(30, 12, 2016);
        ImmutualTime time = new ImmutualTime( 9, 46, 23);
        DateAndTime dateAndTime = new DateAndTime(date, time);
        ImmutualTime actual = dateAndTime.getTime();
        assertEquals(time, actual);
    }

    @Test
    public void getDate2() {
        ImmutualDate date = new ImmutualDate(30, 12, 2016);
        ImmutualTime time = new ImmutualTime( 9, 46, 23);
        DateAndTime dateAndTime = new DateAndTime(date, time);
        ImmutualDate actual = dateAndTime.getDate();
        assertEquals(date, actual);
    }

    @Test
    public void toStringTest() {
        ImmutualDate date = new ImmutualDate(30, 12, 2016);
        ImmutualTime time = new ImmutualTime( 9, 46, 23);
        DateAndTime dateAndTime = new DateAndTime(date, time);
        String actual = dateAndTime.toString();
        String expected = "30.12.2016, 09:46:23";
        assertEquals(expected, actual);
    }

    @Test
    public void toStringWithDelimiterTest() {
        ImmutualDate date = new ImmutualDate(30, 12, 2016);
        ImmutualTime time = new ImmutualTime( 9, 46, 23);
        DateAndTime dateAndTime = new DateAndTime(date, time);
        String delimiter = "###";
        String actual = dateAndTime.toString(delimiter);
        String expected = "30.12.2016" + delimiter + "09:46:23";
        assertEquals(expected, actual);
    }

    @Test
    public void equalsTest() {
        ImmutualDate date = new ImmutualDate(30, 12, 2016);
        ImmutualTime time = new ImmutualTime( 9, 46, 23);
        DateAndTime dateAndTime1 = new DateAndTime(date, time);
        DateAndTime dateAndTime2 = new DateAndTime(date, time);
        assertEquals(dateAndTime1, dateAndTime2);
    }

    @Test
    public void constructFromMillies() {
        long millies = 1490622889912L;
        DateAndTime dateAndTime = new DateAndTime(millies);
        String actual = dateAndTime.toString();
        String expected = "27.03.2017, 15:54:49";
        assertEquals(expected, actual);
    }

    @Test
    public void asYyyyMmDdDelimiterHhMmSs() {
        ImmutualDate date = new ImmutualDate(30, 12, 2016);
        ImmutualTime time = new ImmutualTime( 9, 46, 23);
        DateAndTime dateAndTime = new DateAndTime(date, time);
        String delimiter = "_";
        String actual = dateAndTime.asYyyyMmDdDelimiterHhMmSs(delimiter);
        String expected = "20161230" + delimiter + "094623";
        assertEquals(expected, actual);
    }

    @Test
    public void addSeconds() {
        ImmutualDate date = new ImmutualDate(27,  3, 2017);
        ImmutualTime time = new ImmutualTime(16, 58, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addSeconds(13);

        ImmutualDate newDate = new ImmutualDate(27,  3, 2017);
        ImmutualTime newTime = new ImmutualTime(16, 58, 36);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addSecondsOverMidnigth() {
        ImmutualDate date = new ImmutualDate(27,  3, 2017);
        ImmutualTime time = new ImmutualTime(23, 59, 45);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addSeconds(30);

        ImmutualDate newDate = new ImmutualDate(28,  3, 2017);
        ImmutualTime newTime = new ImmutualTime( 0,  0, 15);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void subtractSecondsOverMidnigth() {
        ImmutualDate date = new ImmutualDate(28,  3, 2017);
        ImmutualTime time = new ImmutualTime( 0,  0, 15);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addSeconds(-30);

        ImmutualDate newDate = new ImmutualDate(27,  3, 2017);
        ImmutualTime newTime = new ImmutualTime(23, 59, 45);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addMinutes() {
        ImmutualDate date = new ImmutualDate(27,  3, 2017);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addMinutes(13);

        ImmutualDate newDate = new ImmutualDate(27, 3, 2017);
        ImmutualTime newTime = new ImmutualTime(17, 17, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addMinutesOverMidnigth() {
        ImmutualDate date = new ImmutualDate(27, 3, 2017);
        ImmutualTime time = new ImmutualTime(23, 59, 45);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addMinutes(1);

        ImmutualDate newDate = new ImmutualDate(28, 3, 2017);
        ImmutualTime newTime = new ImmutualTime( 0,  0, 45);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void subtractMinutesOverMidnigth() {
        ImmutualDate date = new ImmutualDate(28, 3, 2017);
        ImmutualTime time = new ImmutualTime( 0,  20, 15);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addMinutes(-30);

        ImmutualDate newDate = new ImmutualDate(27, 3, 2017);
        ImmutualTime newTime = new ImmutualTime(23, 50, 15);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addHours() {
        ImmutualDate date = new ImmutualDate(27,  3, 2017);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addHours(3);

        ImmutualDate newDate = new ImmutualDate(27,  3, 2017);
        ImmutualTime newTime = new ImmutualTime(20,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addHoursOverMidnigth() {
        ImmutualDate date = new ImmutualDate(27,  3, 2017);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addHours(12);

        ImmutualDate newDate = new ImmutualDate(28,  3, 2017);
        ImmutualTime newTime = new ImmutualTime( 5,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void subtractHoursOverMidnigth() {
        ImmutualDate date = new ImmutualDate(28,  3, 2017);
        ImmutualTime time = new ImmutualTime( 5,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addHours(-12);

        ImmutualDate newDate = new ImmutualDate(27,  3, 2017);
        ImmutualTime newTime = new ImmutualTime(17,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addDays() {
        ImmutualDate date = new ImmutualDate(27,  3, 2017);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addDays(3);

        ImmutualDate newDate = new ImmutualDate(30,  3, 2017);
        ImmutualTime newTime = new ImmutualTime(17,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addDaysOverMonth() {
        ImmutualDate date = new ImmutualDate(27,  3, 2017);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addDays(5);

        ImmutualDate newDate = new ImmutualDate( 1,  4, 2017);
        ImmutualTime newTime = new ImmutualTime(17,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void subtractDaysOverMonth() {
        ImmutualDate date = new ImmutualDate( 1,  4, 2017);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addDays(-5);

        ImmutualDate newDate = new ImmutualDate(27,  3, 2017);
        ImmutualTime newTime = new ImmutualTime(17,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    // months

    @Test
    public void addOneMonth() {
        ImmutualDate date = new ImmutualDate(27,  3, 2017);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addMonths(1);

        ImmutualDate newDate = new ImmutualDate(27,  4, 2017);
        ImmutualTime newTime = new ImmutualTime(17,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addOneMonthApril() {
        ImmutualDate date = new ImmutualDate(27,  4, 2017);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addMonths(1);

        ImmutualDate newDate = new ImmutualDate(27,  5, 2017);
        ImmutualTime newTime = new ImmutualTime(17,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addOneMonthMay() {
        ImmutualDate date = new ImmutualDate(27,  5, 2017);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addMonths(1);

        ImmutualDate newDate = new ImmutualDate(27,  6, 2017);
        ImmutualTime newTime = new ImmutualTime(17,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addTwoMonths() {
        ImmutualDate date = new ImmutualDate(27,  4, 2017);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addMonths(2);

        ImmutualDate newDate = new ImmutualDate(27,  6, 2017);
        ImmutualTime newTime = new ImmutualTime(17,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addThreeMonths() {
        ImmutualDate date = new ImmutualDate(27,  3, 2017);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addMonths(3);

        ImmutualDate newDate = new ImmutualDate(27,  6, 2017);
        ImmutualTime newTime = new ImmutualTime(17,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addMonthsOverYear() {
        ImmutualDate date = new ImmutualDate(27, 12, 2016);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addMonths(1);

        ImmutualDate newDate = new ImmutualDate(27,  1, 2017);
        ImmutualTime newTime = new ImmutualTime(17,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void subtractMonthsOverYear() {
        ImmutualDate date = new ImmutualDate(27,  1, 2017);
        ImmutualTime time = new ImmutualTime(17,  4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addMonths(-1);

        ImmutualDate newDate = new ImmutualDate(27, 12, 2016);
        ImmutualTime newTime = new ImmutualTime(17,  4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void addYears() {
        ImmutualDate date = new ImmutualDate(27, 3, 2017);
        ImmutualTime time = new ImmutualTime(17, 4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addYears(13);

        ImmutualDate newDate = new ImmutualDate(27, 3, 2030);
        ImmutualTime newTime = new ImmutualTime(17, 4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void subtractYears() {
        ImmutualDate date = new ImmutualDate(27, 3, 2030);
        ImmutualTime time = new ImmutualTime(17, 4, 23);
        DateAndTime actual = new DateAndTime(date, time);

        actual = actual.addYears(-13);

        ImmutualDate newDate = new ImmutualDate(27, 3, 2017);
        ImmutualTime newTime = new ImmutualTime(17, 4, 23);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void liesBetweeen() {
        ImmutualDate startDate = new ImmutualDate(27, 3, 2030);
        ImmutualTime startTime = new ImmutualTime(17, 4, 23);
        DateAndTime start = new DateAndTime(startDate, startTime);

        ImmutualDate endDate = new ImmutualDate(29, 3, 2030);
        ImmutualTime endTime = new ImmutualTime( 3, 4, 23);
        DateAndTime end = new DateAndTime(endDate, endTime);

        ImmutualDate betweenDate = new ImmutualDate(28, 3, 2030);
        ImmutualTime betweenTime = new ImmutualTime(20, 4, 23);
        DateAndTime between = new DateAndTime(betweenDate, betweenTime);

        assertTrue(between.liesBetween(start, end));
    }

    @Test
    public void startLiesBetweeen() {
        ImmutualDate startDate = new ImmutualDate(27, 3, 2030);
        ImmutualTime startTime = new ImmutualTime(17, 4, 23);
        DateAndTime start = new DateAndTime(startDate, startTime);

        ImmutualDate endDate = new ImmutualDate(29, 3, 2030);
        ImmutualTime endTime = new ImmutualTime( 3, 4, 23);
        DateAndTime end = new DateAndTime(endDate, endTime);

        assertTrue(start.liesBetween(start, end));
    }

    @Test
    public void endLiesBetweeen() {
        ImmutualDate startDate = new ImmutualDate(27, 3, 2030);
        ImmutualTime startTime = new ImmutualTime(17, 4, 23);
        DateAndTime start = new DateAndTime(startDate, startTime);

        ImmutualDate endDate = new ImmutualDate(29, 3, 2030);
        ImmutualTime endTime = new ImmutualTime( 3, 4, 23);
        DateAndTime end = new DateAndTime(endDate, endTime);

        assertTrue(end.liesBetween(start, end));
    }

    @Test
    public void liesNotBetweeen() {
        ImmutualDate startDate = new ImmutualDate(27, 3, 2030);
        ImmutualTime startTime = new ImmutualTime(17, 4, 23);
        DateAndTime start = new DateAndTime(startDate, startTime);

        ImmutualDate endDate = new ImmutualDate(29, 3, 2030);
        ImmutualTime endTime = new ImmutualTime( 3, 4, 23);
        DateAndTime end = new DateAndTime(endDate, endTime);

        ImmutualDate betweenDate = new ImmutualDate(23, 3, 2030);
        ImmutualTime betweenTime = new ImmutualTime( 3, 4, 24);
        DateAndTime between = new DateAndTime(betweenDate, betweenTime);

        assertFalse(between.liesBetween(start, end));
    }

    @Test
    public void beforeAndAfter1() {
        ImmutualDate startDate = new ImmutualDate(27, 3, 2030);
        ImmutualTime startTime = new ImmutualTime(17, 4, 23);
        DateAndTime start = new DateAndTime(startDate, startTime);

        ImmutualDate endDate = new ImmutualDate(29, 3, 2030);
        ImmutualTime endTime = new ImmutualTime( 3, 4, 23);
        DateAndTime end = new DateAndTime(endDate, endTime);

        assertTrue(start.before(end));
        assertTrue(end.after(start));
        assertFalse(end.before(start));
        assertFalse(start.after(end));
        assertFalse(start.before(start));
        assertFalse(end.after(end));
    }

    @Test
    public void beforeAndAfter2() {
        ImmutualDate startDate = new ImmutualDate(27, 3, 2017);
        ImmutualTime startTime = new ImmutualTime(17, 4, 23);
        DateAndTime start = new DateAndTime(startDate, startTime);

        ImmutualDate endDate = new ImmutualDate(27, 3, 2017);
        ImmutualTime endTime = new ImmutualTime(17, 4, 24);
        DateAndTime end = new DateAndTime(endDate, endTime);

        assertTrue(start.before(end));
        assertTrue(end.after(start));
        assertFalse(end.before(start));
        assertFalse(start.after(end));
        assertFalse(start.before(start));
        assertFalse(end.after(end));
    }

    @Test
    public void differenceEqualDateTime() {
        ImmutualDate date1 = new ImmutualDate(4, 1, 2012);
        ImmutualTime time1 = new ImmutualTime();
        DateAndTime dateAndTime1 = new DateAndTime(date1, time1);
        long difference = dateAndTime1.difference(dateAndTime1);
        assertEquals(0, difference);
    }

    @Test
    public void differenceDifferentDaysEqualTimePositive() {
        ImmutualDate date1 = new ImmutualDate(4, 1, 2012);
        ImmutualTime time1 = new ImmutualTime();
        DateAndTime dateAndTime1 = new DateAndTime(date1, time1);

        ImmutualDate date2 = new ImmutualDate(17, 1, 2012);
        ImmutualTime time2 = new ImmutualTime();
        DateAndTime dateAndTime2 = new DateAndTime(date2, time2);

        long difference = dateAndTime1.difference(dateAndTime2);
        assertEquals(13 * SECONDS_OF_ONE_DAY, difference);
    }

    @Test
    public void differenceDifferentDaysEqualTimeNegative() {
        ImmutualDate date1 = new ImmutualDate(4, 1, 2012);
        ImmutualTime time1 = new ImmutualTime();
        DateAndTime dateAndTime1 = new DateAndTime(date1, time1);

        ImmutualDate date2 = new ImmutualDate(17, 1, 2012);
        ImmutualTime time2 = new ImmutualTime();
        DateAndTime dateAndTime2 = new DateAndTime(date2, time2);

        long difference = dateAndTime2.difference(dateAndTime1);
        assertEquals(-13 * SECONDS_OF_ONE_DAY, difference);
    }

    @Test
    public void differenceEqualDaysDifferentTimePositive() {
        ImmutualDate date1 = new ImmutualDate(4, 1, 2012);
        ImmutualTime time1 = new ImmutualTime(10, 0, 0);
        DateAndTime dateAndTime1 = new DateAndTime(date1, time1);

        ImmutualDate date2 = new ImmutualDate(4, 1, 2012);
        ImmutualTime time2 = new ImmutualTime(10, 0, 17);
        DateAndTime dateAndTime2 = new DateAndTime(date2, time2);

        long difference = dateAndTime1.difference(dateAndTime2);
        assertEquals(17, difference);
    }

    @Test
    public void differenceEqualDaysDifferentTimeNegative() {
        ImmutualDate date1 = new ImmutualDate(4, 1, 2012);
        ImmutualTime time1 = new ImmutualTime(10, 0, 0);
        DateAndTime dateAndTime1 = new DateAndTime(date1, time1);

        ImmutualDate date2 = new ImmutualDate(4, 1, 2012);
        ImmutualTime time2 = new ImmutualTime(10, 0, 17);
        DateAndTime dateAndTime2 = new DateAndTime(date2, time2);

        long difference = dateAndTime2.difference(dateAndTime1);
        assertEquals(-17, difference);
    }

    @Test
    public void differenceDifferentDaysDifferentTimePositive() {
        ImmutualDate date1 = new ImmutualDate(4, 1, 2012);
        ImmutualTime time1 = new ImmutualTime(10, 0, 0);
        DateAndTime dateAndTime1 = new DateAndTime(date1, time1);

        ImmutualDate date2 = new ImmutualDate(17, 1, 2012);
        ImmutualTime time2 = new ImmutualTime(10, 0, 17);
        DateAndTime dateAndTime2 = new DateAndTime(date2, time2);

        long difference = dateAndTime1.difference(dateAndTime2);
        assertEquals(13 * SECONDS_OF_ONE_DAY + 17, difference);
    }

    @Test
    public void differenceDifferentDaysDifferentTimeNegative() {
        ImmutualDate date1 = new ImmutualDate(4, 1, 2012);
        ImmutualTime time1 = new ImmutualTime(10, 0, 0);
        DateAndTime dateAndTime1 = new DateAndTime(date1, time1);

        ImmutualDate date2 = new ImmutualDate(17, 1, 2012);
        ImmutualTime time2 = new ImmutualTime(10, 0, 17);
        DateAndTime dateAndTime2 = new DateAndTime(date2, time2);

        long difference = dateAndTime2.difference(dateAndTime1);
        assertEquals(-13 * SECONDS_OF_ONE_DAY - 17, difference);
    }

    @Test
    public void setSecondsToZero() {
        ImmutualDate date1 = new ImmutualDate(20, 1, 2020);
        ImmutualTime time1 = new ImmutualTime(11, 23, 54);
        DateAndTime actual = new DateAndTime(date1, time1);
        actual = actual.setSecondsToZero();

        ImmutualDate newDate = new ImmutualDate(20, 1, 2020);
        ImmutualTime newTime = new ImmutualTime(11, 23,  0);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void setSecondsToZeroWithZeroSeconds() {
        ImmutualDate date1 = new ImmutualDate(20, 1, 2020);
        ImmutualTime time1 = new ImmutualTime(11, 23,  0);
        DateAndTime actual = new DateAndTime(date1, time1);
        actual = actual.setSecondsToZero();

        ImmutualDate newDate = new ImmutualDate(20, 1, 2020);
        ImmutualTime newTime = new ImmutualTime(11, 23,  0);
        DateAndTime expected = new DateAndTime(newDate, newTime);
        assertEquals(expected, actual);
    }

    @Test
    public void toEpoch() {
        long millis = 1490622889000L;
        DateAndTime dateAndTime = new DateAndTime(millis);
        String actual = dateAndTime.toString();
        String expected = "27.03.2017, 15:54:49";
        assertEquals(expected, actual);

        long actualMillis = dateAndTime.toEpoch();
        assertEquals(millis, actualMillis);
    }

}
