package de.duehl.basics.datetime.date;

/*
 * Copyright 2026 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.Calendar;
import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;

public class DateHelperTest {

    @Test
    public void calendarToDateTestNow() {
        Calendar cal = Calendar.getInstance();
        ImmutualDate date = DateHelper.calendarToDate(cal);
        assertNotNull(date);
    }

    @Test
    public void calendarToDateTestConcreteDate() {
        Calendar cal = Calendar.getInstance();
        cal.set(2016, 10 - 1, 12, 11, 25, 0);

        ImmutualDate date = DateHelper.calendarToDate(cal);

        int actualYear = date.getYear();
        int actualMonth = date.getMonth();
        int actualDay = date.getDay();

        int expectedYear = 2016;
        int expectedMonth = 10;
        int expectedDay = 12;

        assertEquals(expectedYear, actualYear);
        assertEquals(expectedMonth, actualMonth);
        assertEquals(expectedDay, actualDay);
    }

    @Test
    public void getDateFromYYYYMMDD() {
        String yearMonthDay = "20211004";
        ImmutualDate actual = DateHelper.getDateFromYYYYMMDD(yearMonthDay);
        ImmutualDate expected = new ImmutualDate(4, 10, 2021);
        assertEquals(expected, actual);
    }

    @Test
    public void getDateFromDDMMYYYY() {
        String yearMonthDay = "03022022";
        ImmutualDate actual = DateHelper.getDateFromDDMMYYYY(yearMonthDay);
        ImmutualDate expected = new ImmutualDate(3, 2, 2022);
        assertEquals(expected, actual);
    }

    @Test
    public void isInDdDotMmDotYyyyFormat() {
        String yearMonthDay = "11.08.2025";
        assertTrue(DateHelper.isInDdDotMmDotYyyyFormat(yearMonthDay));
    }

    @Test
    public void isInDdDotMmDotYyyyFormatThreeDots() {
        String yearMonthDay = "11.08.20.25";
        assertFalse(DateHelper.isInDdDotMmDotYyyyFormat(yearMonthDay));
    }

    @Test
    public void isInDdDotMmDotYyyyFormatKomma() {
        String yearMonthDay = "11.08,2025";
        assertFalse(DateHelper.isInDdDotMmDotYyyyFormat(yearMonthDay));
    }

    @Test
    public void isInDdDotMmDotYyyyFormatWrongYearLength() {
        String yearMonthDay = "11.08.225";
        assertFalse(DateHelper.isInDdDotMmDotYyyyFormat(yearMonthDay));
    }

    @Test
    public void isInDdDotMmDotYyyyFormatEmpty() {
        String yearMonthDay = "";
        assertFalse(DateHelper.isInDdDotMmDotYyyyFormat(yearMonthDay));
    }

    @Test
    public void sortFromOldToNew() {
        ImmutualDate date1 = new ImmutualDate(29, 2, 2012);
        ImmutualDate date2 = new ImmutualDate(13, 5, 2010);
        ImmutualDate date3 = new ImmutualDate(28, 7, 2021);
        ImmutualDate date4 = new ImmutualDate(2, 2, 2001);
        ImmutualDate date5 = new ImmutualDate(4, 10, 2021);
        ImmutualDate date6 = new ImmutualDate(3, 10, 2021);
        ImmutualDate date7 = new ImmutualDate(5, 3, 2001);
        List<ImmutualDate> dates = CollectionsHelper.buildListFrom(date1, date2, date3, date4,
                date5, date6, date7);
        assertEquals(date1, dates.get(0));
        assertEquals(date2, dates.get(1));
        assertEquals(date3, dates.get(2));
        assertEquals(date4, dates.get(3));
        assertEquals(date5, dates.get(4));
        assertEquals(date6, dates.get(5));
        assertEquals(date7, dates.get(6));
        DateHelper.sortFromOldToNew(dates);
        assertEquals(date4, dates.get(0));
        assertEquals(date7, dates.get(1));
        assertEquals(date2, dates.get(2));
        assertEquals(date1, dates.get(3));
        assertEquals(date3, dates.get(4));
        assertEquals(date6, dates.get(5));
        assertEquals(date5, dates.get(6));
    }

    @Test
    public void sortFromNewToOld() {
        ImmutualDate date1 = new ImmutualDate(29, 2, 2012);
        ImmutualDate date2 = new ImmutualDate(13, 5, 2010);
        ImmutualDate date3 = new ImmutualDate(28, 7, 2021);
        ImmutualDate date4 = new ImmutualDate(2, 2, 2001);
        ImmutualDate date5 = new ImmutualDate(4, 10, 2021);
        ImmutualDate date6 = new ImmutualDate(3, 10, 2021);
        ImmutualDate date7 = new ImmutualDate(5, 3, 2001);
        List<ImmutualDate> dates = CollectionsHelper.buildListFrom(date1, date2, date3, date4,
                date5, date6, date7);
        assertEquals(date1, dates.get(0));
        assertEquals(date2, dates.get(1));
        assertEquals(date3, dates.get(2));
        assertEquals(date4, dates.get(3));
        assertEquals(date5, dates.get(4));
        assertEquals(date6, dates.get(5));
        assertEquals(date7, dates.get(6));
        DateHelper.sortFromNewToOld(dates);
        assertEquals(date5, dates.get(0));
        assertEquals(date6, dates.get(1));
        assertEquals(date3, dates.get(2));
        assertEquals(date1, dates.get(3));
        assertEquals(date2, dates.get(4));
        assertEquals(date7, dates.get(5));
        assertEquals(date4, dates.get(6));
    }

    @Test
    public void getRange() {
        ImmutualDate minimalDate = new ImmutualDate(22, 11, 2021);
        ImmutualDate maximalDate = new ImmutualDate(26, 11, 2021);
        List<ImmutualDate> range = DateHelper.getRange(minimalDate, maximalDate);
        assertEquals(5, range.size());
        assertEquals(new ImmutualDate(22, 11, 2021), range.get(0));
        assertEquals(new ImmutualDate(23, 11, 2021), range.get(1));
        assertEquals(new ImmutualDate(24, 11, 2021), range.get(2));
        assertEquals(new ImmutualDate(25, 11, 2021), range.get(3));
        assertEquals(new ImmutualDate(26, 11, 2021), range.get(4));
    }

    @Test
    public void getRangeWithEqualMinAndMaxDate() {
        ImmutualDate minimalDate = new ImmutualDate(22, 11, 2021);
        ImmutualDate maximalDate = new ImmutualDate(22, 11, 2021);
        List<ImmutualDate> range = DateHelper.getRange(minimalDate, maximalDate);
        assertEquals(1, range.size());
        assertEquals(new ImmutualDate(22, 11, 2021), range.get(0));
    }

    @Test (expected = IllegalArgumentException.class)
    public void getRangeFailWithMinAfterMaxDate() {
        ImmutualDate minimalDate = new ImmutualDate(26, 11, 2021);
        ImmutualDate maximalDate = new ImmutualDate(22, 11, 2021);
        DateHelper.getRange(minimalDate, maximalDate);
    }

    @Test
    public void getDatesOfMonthWithDate() {
        ImmutualDate date = new ImmutualDate(29, 11, 2021);
        List<ImmutualDate> actualDates = DateHelper.getDatesOfMonthWithDate(date);

        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                new ImmutualDate( 1, 11, 2021), // Montag
                new ImmutualDate( 2, 11, 2021),
                new ImmutualDate( 3, 11, 2021),
                new ImmutualDate( 4, 11, 2021),
                new ImmutualDate( 5, 11, 2021),
                new ImmutualDate( 6, 11, 2021),
                new ImmutualDate( 7, 11, 2021),
                new ImmutualDate( 8, 11, 2021), // Montag
                new ImmutualDate( 9, 11, 2021),
                new ImmutualDate(10, 11, 2021),
                new ImmutualDate(11, 11, 2021),
                new ImmutualDate(12, 11, 2021),
                new ImmutualDate(13, 11, 2021),
                new ImmutualDate(14, 11, 2021),
                new ImmutualDate(15, 11, 2021), // Montag
                new ImmutualDate(16, 11, 2021),
                new ImmutualDate(17, 11, 2021),
                new ImmutualDate(18, 11, 2021),
                new ImmutualDate(19, 11, 2021),
                new ImmutualDate(20, 11, 2021),
                new ImmutualDate(21, 11, 2021),
                new ImmutualDate(22, 11, 2021), // Montag
                new ImmutualDate(23, 11, 2021),
                new ImmutualDate(24, 11, 2021),
                new ImmutualDate(25, 11, 2021),
                new ImmutualDate(26, 11, 2021),
                new ImmutualDate(27, 11, 2021),
                new ImmutualDate(28, 11, 2021),
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021)
                );
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void getDatesOfMonthWithDateAtFirstDayOfMonth() {
        ImmutualDate date = new ImmutualDate( 1, 11, 2021);
        List<ImmutualDate> actualDates = DateHelper.getDatesOfMonthWithDate(date);

        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                new ImmutualDate( 1, 11, 2021), // Montag
                new ImmutualDate( 2, 11, 2021),
                new ImmutualDate( 3, 11, 2021),
                new ImmutualDate( 4, 11, 2021),
                new ImmutualDate( 5, 11, 2021),
                new ImmutualDate( 6, 11, 2021),
                new ImmutualDate( 7, 11, 2021),
                new ImmutualDate( 8, 11, 2021), // Montag
                new ImmutualDate( 9, 11, 2021),
                new ImmutualDate(10, 11, 2021),
                new ImmutualDate(11, 11, 2021),
                new ImmutualDate(12, 11, 2021),
                new ImmutualDate(13, 11, 2021),
                new ImmutualDate(14, 11, 2021),
                new ImmutualDate(15, 11, 2021), // Montag
                new ImmutualDate(16, 11, 2021),
                new ImmutualDate(17, 11, 2021),
                new ImmutualDate(18, 11, 2021),
                new ImmutualDate(19, 11, 2021),
                new ImmutualDate(20, 11, 2021),
                new ImmutualDate(21, 11, 2021),
                new ImmutualDate(22, 11, 2021), // Montag
                new ImmutualDate(23, 11, 2021),
                new ImmutualDate(24, 11, 2021),
                new ImmutualDate(25, 11, 2021),
                new ImmutualDate(26, 11, 2021),
                new ImmutualDate(27, 11, 2021),
                new ImmutualDate(28, 11, 2021),
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021)
                );
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void getDatesOfMonthWithDateAtLastDayOfMonth() {
        ImmutualDate date = new ImmutualDate(30, 11, 2021);
        List<ImmutualDate> actualDates = DateHelper.getDatesOfMonthWithDate(date);

        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                new ImmutualDate( 1, 11, 2021), // Montag
                new ImmutualDate( 2, 11, 2021),
                new ImmutualDate( 3, 11, 2021),
                new ImmutualDate( 4, 11, 2021),
                new ImmutualDate( 5, 11, 2021),
                new ImmutualDate( 6, 11, 2021),
                new ImmutualDate( 7, 11, 2021),
                new ImmutualDate( 8, 11, 2021), // Montag
                new ImmutualDate( 9, 11, 2021),
                new ImmutualDate(10, 11, 2021),
                new ImmutualDate(11, 11, 2021),
                new ImmutualDate(12, 11, 2021),
                new ImmutualDate(13, 11, 2021),
                new ImmutualDate(14, 11, 2021),
                new ImmutualDate(15, 11, 2021), // Montag
                new ImmutualDate(16, 11, 2021),
                new ImmutualDate(17, 11, 2021),
                new ImmutualDate(18, 11, 2021),
                new ImmutualDate(19, 11, 2021),
                new ImmutualDate(20, 11, 2021),
                new ImmutualDate(21, 11, 2021),
                new ImmutualDate(22, 11, 2021), // Montag
                new ImmutualDate(23, 11, 2021),
                new ImmutualDate(24, 11, 2021),
                new ImmutualDate(25, 11, 2021),
                new ImmutualDate(26, 11, 2021),
                new ImmutualDate(27, 11, 2021),
                new ImmutualDate(28, 11, 2021),
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021)
                );
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void getDatesOfWeekWithDateStartingWithMonday() {
        ImmutualDate date = new ImmutualDate(29, 11, 2021);
        List<ImmutualDate> actualDates = DateHelper.getDatesOfWeekWithDate(date);

        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021),
                new ImmutualDate( 1, 12, 2021),
                new ImmutualDate( 2, 12, 2021),
                new ImmutualDate( 3, 12, 2021),
                new ImmutualDate( 4, 12, 2021),
                new ImmutualDate( 5, 12, 2021)
                );
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void getDatesOfWeekWithDateStartingWithTuesday() {
        ImmutualDate date = new ImmutualDate(30, 11, 2021);
        List<ImmutualDate> actualDates = DateHelper.getDatesOfWeekWithDate(date);

        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021),
                new ImmutualDate( 1, 12, 2021),
                new ImmutualDate( 2, 12, 2021),
                new ImmutualDate( 3, 12, 2021),
                new ImmutualDate( 4, 12, 2021),
                new ImmutualDate( 5, 12, 2021)
                );
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void getDatesOfWeekWithDateStartingWithWednesday() {
        ImmutualDate date = new ImmutualDate( 1, 12, 2021);
        List<ImmutualDate> actualDates = DateHelper.getDatesOfWeekWithDate(date);

        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021),
                new ImmutualDate( 1, 12, 2021),
                new ImmutualDate( 2, 12, 2021),
                new ImmutualDate( 3, 12, 2021),
                new ImmutualDate( 4, 12, 2021),
                new ImmutualDate( 5, 12, 2021)
                );
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void getDatesOfWeekWithDateStartingWithThursday() {
        ImmutualDate date = new ImmutualDate( 2, 12, 2021);
        List<ImmutualDate> actualDates = DateHelper.getDatesOfWeekWithDate(date);

        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021),
                new ImmutualDate( 1, 12, 2021),
                new ImmutualDate( 2, 12, 2021),
                new ImmutualDate( 3, 12, 2021),
                new ImmutualDate( 4, 12, 2021),
                new ImmutualDate( 5, 12, 2021)
                );
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void getDatesOfWeekWithDateStartingWithFriday() {
        ImmutualDate date = new ImmutualDate( 3, 12, 2021);
        List<ImmutualDate> actualDates = DateHelper.getDatesOfWeekWithDate(date);

        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021),
                new ImmutualDate( 1, 12, 2021),
                new ImmutualDate( 2, 12, 2021),
                new ImmutualDate( 3, 12, 2021),
                new ImmutualDate( 4, 12, 2021),
                new ImmutualDate( 5, 12, 2021)
                );
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void getDatesOfWeekWithDateStartingWithSaturday() {
        ImmutualDate date = new ImmutualDate( 4, 12, 2021);
        List<ImmutualDate> actualDates = DateHelper.getDatesOfWeekWithDate(date);

        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021),
                new ImmutualDate( 1, 12, 2021),
                new ImmutualDate( 2, 12, 2021),
                new ImmutualDate( 3, 12, 2021),
                new ImmutualDate( 4, 12, 2021),
                new ImmutualDate( 5, 12, 2021)
                );
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void getDatesOfWeekWithDateStartingWithSunday() {
        ImmutualDate date = new ImmutualDate( 5, 12, 2021);
        List<ImmutualDate> actualDates = DateHelper.getDatesOfWeekWithDate(date);

        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021),
                new ImmutualDate( 1, 12, 2021),
                new ImmutualDate( 2, 12, 2021),
                new ImmutualDate( 3, 12, 2021),
                new ImmutualDate( 4, 12, 2021),
                new ImmutualDate( 5, 12, 2021)
                );
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void keepOnlyWorkDays() {
        List<ImmutualDate> dates = CollectionsHelper.buildListFrom(
                new ImmutualDate(22, 11, 2021), // Montag
                new ImmutualDate(23, 11, 2021),
                new ImmutualDate(24, 11, 2021),
                new ImmutualDate(25, 11, 2021),
                new ImmutualDate(26, 11, 2021),
                new ImmutualDate(27, 11, 2021),
                new ImmutualDate(28, 11, 2021),
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021)
                );
        List<ImmutualDate> actualDates = DateHelper.keepOnlyWorkDays(dates);
        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                new ImmutualDate(22, 11, 2021), // Montag
                new ImmutualDate(23, 11, 2021),
                new ImmutualDate(24, 11, 2021),
                new ImmutualDate(25, 11, 2021),
                new ImmutualDate(26, 11, 2021),
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021)
                );
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void keepOnlyWorkDaysFullMonth() {
        List<ImmutualDate> dates = CollectionsHelper.buildListFrom(
                new ImmutualDate( 1, 11, 2021), // Montag aber Feiertag!
                new ImmutualDate( 2, 11, 2021),
                new ImmutualDate( 3, 11, 2021),
                new ImmutualDate( 4, 11, 2021),
                new ImmutualDate( 5, 11, 2021),
                new ImmutualDate( 6, 11, 2021),
                new ImmutualDate( 7, 11, 2021),
                new ImmutualDate( 8, 11, 2021), // Montag
                new ImmutualDate( 9, 11, 2021),
                new ImmutualDate(10, 11, 2021),
                new ImmutualDate(11, 11, 2021),
                new ImmutualDate(12, 11, 2021),
                new ImmutualDate(13, 11, 2021),
                new ImmutualDate(14, 11, 2021),
                new ImmutualDate(15, 11, 2021), // Montag
                new ImmutualDate(16, 11, 2021),
                new ImmutualDate(17, 11, 2021),
                new ImmutualDate(18, 11, 2021),
                new ImmutualDate(19, 11, 2021),
                new ImmutualDate(20, 11, 2021),
                new ImmutualDate(21, 11, 2021),
                new ImmutualDate(22, 11, 2021), // Montag
                new ImmutualDate(23, 11, 2021),
                new ImmutualDate(24, 11, 2021),
                new ImmutualDate(25, 11, 2021),
                new ImmutualDate(26, 11, 2021),
                new ImmutualDate(27, 11, 2021),
                new ImmutualDate(28, 11, 2021),
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021)
                );
        List<ImmutualDate> actualDates = DateHelper.keepOnlyWorkDays(dates);
        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                new ImmutualDate( 2, 11, 2021),
                new ImmutualDate( 3, 11, 2021),
                new ImmutualDate( 4, 11, 2021),
                new ImmutualDate( 5, 11, 2021),
                new ImmutualDate( 8, 11, 2021), // Montag
                new ImmutualDate( 9, 11, 2021),
                new ImmutualDate(10, 11, 2021),
                new ImmutualDate(11, 11, 2021),
                new ImmutualDate(12, 11, 2021),
                new ImmutualDate(15, 11, 2021), // Montag
                new ImmutualDate(16, 11, 2021),
                new ImmutualDate(17, 11, 2021),
                new ImmutualDate(18, 11, 2021),
                new ImmutualDate(19, 11, 2021),
                new ImmutualDate(22, 11, 2021), // Montag
                new ImmutualDate(23, 11, 2021),
                new ImmutualDate(24, 11, 2021),
                new ImmutualDate(25, 11, 2021),
                new ImmutualDate(26, 11, 2021),
                new ImmutualDate(29, 11, 2021), // Montag
                new ImmutualDate(30, 11, 2021)
                );
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void keepOnlyWorkDaysEmptyList() {
        List<ImmutualDate> dates = CollectionsHelper.buildListFrom(
                );
        List<ImmutualDate> expectedDates = CollectionsHelper.buildListFrom(
                );
        List<ImmutualDate> actualDates = DateHelper.keepOnlyWorkDays(dates);
        assertEquals(expectedDates, actualDates);
    }

    @Test
    public void determineNewestDate() {
        String actual = DateHelper.determineNewestDate("20.03.2021", "10.08.2022", "31.12.2020");
        Object expected = "10.08.2022";
        assertEquals(expected , actual);
    }

    @Test
    public void createCorrectDate() {
        String date = "20.10.2022";
        String actual = DateHelper.createCorrectDate(date);
        String expected = "20.10.2022";
        assertEquals(expected , actual);
    }

    @Test
    public void createCorrectDateSmallDay() {
        String date = "2.10.2022";
        String actual = DateHelper.createCorrectDate(date);
        String expected = "02.10.2022";
        assertEquals(expected , actual);
    }

    @Test
    public void createCorrectDateSmallMonth() {
        String date = "20.9.2022";
        String actual = DateHelper.createCorrectDate(date);
        String expected = "20.09.2022";
        assertEquals(expected , actual);
    }

    @Test
    public void createCorrectDateSmallMonthAndDay() {
        String date = "2.9.2022";
        String actual = DateHelper.createCorrectDate(date);
        String expected = "02.09.2022";
        assertEquals(expected , actual);
    }

    @Test
    public void createCorrectDateGermanMonthName() {
        String date = "20. Oktober 2022";
        String actual = DateHelper.createCorrectDate(date);
        String expected = "20.10.2022";
        assertEquals(expected , actual);
    }

    @Test
    public void insertDateDotsDDMMYYYY() {
        String dateWithoutDots = "20102022";
        String actual = DateHelper.insertDateDotsDDMMYYYY(dateWithoutDots);
        String expected = "20.10.2022";
        assertEquals(expected , actual);
    }

    @Test
    public void insertDateDotsDDMMYYYYWrongLength() {
        String dateWithoutDots = "201020229";
        String actual = DateHelper.insertDateDotsDDMMYYYY(dateWithoutDots);
        String expected = dateWithoutDots;
        assertEquals(expected , actual);
    }

    @Test
    public void insertDateDotsDDMMYYYYNotOnlyDigits() {
        String dateWithoutDots = "201o2022";
        String actual = DateHelper.insertDateDotsDDMMYYYY(dateWithoutDots);
        String expected = dateWithoutDots;
        assertEquals(expected , actual);
    }

    @Test
    public void removeDateDotsDDMMYYYY() {
        String dateWithoutDots = "13.12.2022";
        String actual = DateHelper.removeDateDotsDDMMYYYY(dateWithoutDots);
        String expected = "13122022";
        assertEquals(expected , actual);
    }

    @Test
    public void removeDateDotsDDMMYYYYWrongLength() {
        String dateWithoutDots = "13.12.20221";
        String actual = DateHelper.removeDateDotsDDMMYYYY(dateWithoutDots);
        String expected = dateWithoutDots;
        assertEquals(expected , actual);
    }

    @Test
    public void conainesAnyDate() {
        assertTrue(DateHelper.conainesAnyDate("19.11.2025"));
        assertTrue(DateHelper.conainesAnyDate("Text mit 19.11.2025 Datum"));
        assertTrue(DateHelper.conainesAnyDate("Text mit 19. Nov. 2025 Datum"));
        assertTrue(DateHelper.conainesAnyDate("Text mit 19. November 2025 Datum"));

        assertTrue(DateHelper.conainesAnyDate("Text mit November 2025 Datum"));
        assertTrue(DateHelper.conainesAnyDate("Text mit 2025 Datum"));
    }

    @Test
    public void conainesAnyFullDate() {
        assertTrue(DateHelper.conainesAnyFullDate("19.11.2025"));
        assertTrue(DateHelper.conainesAnyFullDate("Text mit 19.11.2025 Datum"));
        assertTrue(DateHelper.conainesAnyFullDate("Text mit 19. Nov. 2025 Datum"));
        assertTrue(DateHelper.conainesAnyFullDate("Text mit 19. November 2025 Datum"));

        assertFalse(DateHelper.conainesAnyFullDate("Text mit November 2025 Datum"));
        assertFalse(DateHelper.conainesAnyFullDate("Text mit 2025 Datum"));
    }

    @Test
    public void isYear() {
        assertTrue(DateHelper.isYear("1967"));
        assertTrue(DateHelper.isYear("2025"));
        assertTrue(DateHelper.isYear("2026"));

        assertFalse(DateHelper.isYear("1865"));
        assertFalse(DateHelper.isYear("2122"));
        assertFalse(DateHelper.isYear("3245"));

        assertFalse(DateHelper.isYear("226"));
        assertFalse(DateHelper.isYear("20026"));
    }

}
