package de.duehl.basics.datetime.date;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.junit.Test;

public class ImmutualDateTest {

    @Test
    public void create2() {
        ImmutualDate date = new ImmutualDate(1, 2, 3);
        assertNotNull(date);
    }

    @Test
    public void create3() {
        ImmutualDate date = new ImmutualDate("foo");
        assertNotNull(date);
    }

    @Test
    public void createFromWrongString() {
        ImmutualDate date = new ImmutualDate("foo");

        assertEquals(0, date.getDay()  );
        assertEquals(0, date.getMonth());
        assertEquals(0, date.getYear() );
    }

    @Test
    public void createFromString01() {
        ImmutualDate date = new ImmutualDate("11. Februar 1967");

        assertEquals(11,   date.getDay()  );
        assertEquals(2,    date.getMonth());
        assertEquals(1967, date.getYear() );
    }

    @Test
    public void createFromString01b() {
        ImmutualDate date = new ImmutualDate("11. Dronember 1967");

        assertEquals(11,   date.getDay()  );
        assertEquals(0,    date.getMonth());
        assertEquals(1967, date.getYear() );

        assertFalse(date.isValid());
    }

    @Test
    public void createFromString02() {
        ImmutualDate date = new ImmutualDate("11.02.1967");

        assertEquals(11,   date.getDay()  );
        assertEquals(2,    date.getMonth());
        assertEquals(1967, date.getYear() );
    }

    @Test
    public void createFromString03() {
        ImmutualDate date = new ImmutualDate("Wed Jan 18 12:01:50 CET 2012");
        // dow mon dd hh:mm:ss zzz yyyy

        assertEquals(18,   date.getDay()  );
        assertEquals(1,    date.getMonth());
        assertEquals(2012, date.getYear() );
    }


    @Test
    public void leapYear001() {
        ImmutualDate date = new ImmutualDate(29, 2, 1899);
        assertFalse(date.isLeapYear());
    }

    @Test
    public void leapYear002() {
        ImmutualDate date = new ImmutualDate(29, 2, 1900);
        assertFalse(date.isLeapYear());
    }

    @Test
    public void leapYear003() {
        ImmutualDate date = new ImmutualDate(29, 2, 1904);
        assertTrue(date.isLeapYear());
    }

    @Test
    public void leapYear004() {
        ImmutualDate date = new ImmutualDate(29, 2, 1908);
        assertTrue(date.isLeapYear());
    }

    @Test
    public void leapYear005() {
        ImmutualDate date = new ImmutualDate(29, 2, 1996);
        assertTrue(date.isLeapYear());
    }

    @Test
    public void leapYear006() {
        ImmutualDate date = new ImmutualDate(29, 2, 2000);
        assertTrue(date.isLeapYear());
    }

    @Test
    public void leapYear007() {
        ImmutualDate date = new ImmutualDate(29, 2, 2001);
        assertFalse(date.isLeapYear());
    }

    @Test
    public void leapYear008() {
        ImmutualDate date = new ImmutualDate(29, 2, 2012);
        assertTrue(date.isLeapYear());
    }

    @Test
    public void validDate001() {
        ImmutualDate date = new ImmutualDate(29, 2, 2011);
        assertFalse(date.isValid());
    }

    @Test
    public void validDate002() {
        ImmutualDate date = new ImmutualDate(29, 2, 2012);
        assertTrue(date.isValid());
    }

    @Test
    public void validDate003() {
        ImmutualDate date = new ImmutualDate(4, 1, 2012);
        assertTrue(date.isValid());
    }

    @Test
    public void validDate004() {
        ImmutualDate date = new ImmutualDate(32, 1, 2012);
        assertFalse(date.isValid());
    }

    @Test
    public void validDate005() {
        assertFalse(new ImmutualDate(31,  4, 2012).isValid());
    }

    @Test
    public void validDate006() {
        assertFalse(new ImmutualDate( 0,  1, 2012).isValid());
    }

    @Test
    public void validDate007() {
        assertFalse(new ImmutualDate( 2,  4,  999).isValid());
    }

    @Test
    public void validDate008() {
        assertFalse(new ImmutualDate( 1,  5,10000).isValid());
    }

    @Test
    public void validDate009() {
        assertFalse(new ImmutualDate(31,  6, 2012).isValid());
    }

    @Test
    public void validDate010() {
        assertFalse(new ImmutualDate(31,  9, 2012).isValid());
    }

    @Test
    public void validDate011() {
        assertFalse(new ImmutualDate(31, 11, 2012).isValid());
    }

    @Test
    public void validDate012() {
        assertFalse(new ImmutualDate(31,  4, 2012).isValid());
    }

    @Test
    public void validDate013() {
        assertFalse(new ImmutualDate(31,  4, 2012).isValid());
    }

    @Test
    public void validDate014() {
        assertFalse(new ImmutualDate(31,  4, 2012).isValid());
    }

    @Test
    public void validDate015() {
        assertFalse(new ImmutualDate(30,  2, 2012).isValid());
    }

    @Test
    public void validDate016() {
        assertFalse(new ImmutualDate(29,  2, 2011).isValid());
    }

    @Test
    public void validDate017() {
        assertFalse(new ImmutualDate("Unsinn").isValid());
    }

    @Test
    public void validDateZero01() {
        assertFalse(new ImmutualDate(29,  2, 2011).isValidWithYearZero());
    }

    @Test
    public void validDateZero02() {
        assertFalse(new ImmutualDate(29,  1, -1).isValidWithYearZero());
    }

    @Test
    public void validDateZero03() {
        assertTrue(new ImmutualDate(29,  1, 0).isValidWithYearZero());
    }

    @Test
    public void calculateDayDifference001() {
        ImmutualDate date1 = new ImmutualDate(4, 1, 2012);
        int difference = date1.calculateDayDifference(date1);
        assertEquals(0, difference);
    }

    @Test
    public void calculateDayDifference002() {
        ImmutualDate date1 = new ImmutualDate(4, 1, 2012);
        ImmutualDate date2 = new ImmutualDate(17, 1, 2012);
        int difference = date1.calculateDayDifference(date2);
        assertEquals(13, difference);
    }

    @Test
    public void calculateDayDifference003() {
        ImmutualDate date1 = new ImmutualDate(4, 1, 2012);
        ImmutualDate date2 = new ImmutualDate(17, 1, 2012);
        int difference = date2.calculateDayDifference(date1);
        assertEquals(-13, difference);
    }

    @Test
    public void calculateDayDifference004() {
        ImmutualDate date1 = new ImmutualDate(4, 1, 2011);
        ImmutualDate date2 = new ImmutualDate(4, 1, 2012);
        int difference = date1.calculateDayDifference(date2);
        assertEquals(365, difference);
    }

    @Test
    public void calculateDayDifference005() {
        ImmutualDate date1 = new ImmutualDate(4, 1, 2012);
        ImmutualDate date2 = new ImmutualDate(4, 1, 2013);
        int difference = date1.calculateDayDifference(date2);
        assertEquals(366, difference);
    }

    @Test
    public void addDays001() {
        ImmutualDate date = new ImmutualDate(4, 1, 2012);
        date = date.addDays(1);

        assertEquals(5,    date.getDay()  );
        assertEquals(1,    date.getMonth());
        assertEquals(2012, date.getYear() );
    }

    @Test
    public void addDays002() {
        ImmutualDate date = new ImmutualDate(1, 5, 2012);
        date = date.addDays(-1);

        assertEquals(30,   date.getDay()  );
        assertEquals(4,    date.getMonth());
        assertEquals(2012, date.getYear() );
    }

    @Test
    public void addDays003() {
        ImmutualDate date = new ImmutualDate(1, 5, 2012);
        date = date.addDays(-4);

        assertEquals(27,   date.getDay()  );
        assertEquals(4,    date.getMonth());
        assertEquals(2012, date.getYear() );
    }

    @Test
    public void addDays004() {
        ImmutualDate date = new ImmutualDate(31, 5, 2012);
        date = date.addDays(1);

        assertEquals(1,    date.getDay()  );
        assertEquals(6,    date.getMonth());
        assertEquals(2012, date.getYear() );
    }

    @Test
    public void addDays005() {
        ImmutualDate date = new ImmutualDate(1, 1, 2011);
        date = date.addDays(365);

        assertEquals(1,    date.getDay()  );
        assertEquals(1,    date.getMonth());
        assertEquals(2012, date.getYear() );
    }

    @Test
    public void addDays006() {
        ImmutualDate date = new ImmutualDate(1, 1, 2012);
        date = date.addDays(365);

        assertEquals(31,   date.getDay()  );
        assertEquals(12,   date.getMonth());
        assertEquals(2012, date.getYear() );
    }

    @Test
    public void addDaysAndCalculate() {
        /* Hier wird getestet, ob addDays und calculateDayDifference zueinander
         * passen, indem für alle validen Datumswerte zwischen dem 1.1.1895 und
         * dem 31.12.2020 das Datum im Abstand von -400 bis +400 berechnet
         * werden und dann vergleichen wird, ob die Abstandsberechnung den
         * gleichen Abstand liefert.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                for (int day=1; day<=31; ++day) {
                    ImmutualDate date1 = new ImmutualDate(day, month, year);
                    if (date1.isValid()) {
                        for (int abstand=-400; abstand<=400; ++abstand) {
                            ImmutualDate date2 = date1.addDays(abstand);
                            int abstand2 = date1.calculateDayDifference(date2);
                            assertEquals(abstand, abstand2);
                        }
                    }
                }
            }
        }
    }

    @Test
    public void mupltipleDaysValid1() {
        /* Hier wird getestet, ob alle Tage zwischen 1 und 28 für alle
         * Datumswerte zwischen dem 1.1.1895 und dem 28.12.2020 valide sind.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                for (int day=1; day<=28; ++day) {
                    ImmutualDate date = new ImmutualDate(day, month, year);
                    assertTrue(date.isValid());
                }
            }
        }
    }

    @Test
    public void mupltipleDaysValid2() {
        /* Hier wird getestet, ob alle 29. und 30. für alle Datumswerte
         * zwischen dem 29.1.1895 und dem 30.12.2020 mit Ausnahme des Februars
         * valide sind.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                if (month != 2) {
                    for (int day=29; day<=30; ++day) {
                        ImmutualDate date = new ImmutualDate(day, month, year);
                        assertTrue(date.isValid());
                    }
                }
            }
        }
    }

    @Test
    public void mupltipleDaysValid3() {
        /* Hier wird getestet, ob alle 31. für alle passenden Monate
         * zwischen dem 31.1.1895 und dem 31.12.2020 valide sind.
         */
        int day = 31;
        for (int year=1895; year<=2020; ++year) {
            ImmutualDate date1  = new ImmutualDate(day,  1, year);
            ImmutualDate date3  = new ImmutualDate(day,  3, year);
            ImmutualDate date5  = new ImmutualDate(day,  5, year);
            ImmutualDate date7  = new ImmutualDate(day,  7, year);
            ImmutualDate date8  = new ImmutualDate(day,  8, year);
            ImmutualDate date10 = new ImmutualDate(day, 10, year);
            ImmutualDate date12 = new ImmutualDate(day, 12, year);
            assertTrue(date1.isValid());
            assertTrue(date3.isValid());
            assertTrue(date5.isValid());
            assertTrue(date7.isValid());
            assertTrue(date8.isValid());
            assertTrue(date10.isValid());
            assertTrue(date12.isValid());
        }
    }

    @Test
    public void mupltipleDaysInvalid1() {
        /* Hier wird getestet, ob alle Tage zwischen -4. und 0. für alle
         * Datumswerte zwischen dem Januar 1895 und dem Dezember 2020 invalide
         * sind.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                for (int day=-4; day<=0; ++day) {
                    ImmutualDate date = new ImmutualDate(day, month, year);
                    assertFalse(date.isValid());
                }
            }
        }
    }

    @Test
    public void mupltipleDaysInvalid2() {
        /* Hier wird getestet, ob alle Tage zwischen 32. und 45. für alle
         * Datumswerte zwischen dem Januar 1895 und dem Dezember 2020 invalide
         * sind.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                for (int day=32; day<=45; ++day) {
                    ImmutualDate date = new ImmutualDate(day, month, year);
                    assertFalse(date.isValid());
                }
            }
        }
    }


    @Test
    public void dayOfTheWeek001() {
        ImmutualDate date = new ImmutualDate(14, 7, 1789); // Erstürmung der Bastille
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.TUESDAY, weekday);
    }

    @Test
    public void dayOfTheWeek002() {
        ImmutualDate date = new ImmutualDate(23, 5, 1949); // Gründung der Bundesrepublik
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.MONDAY, weekday);
    }

    @Test
    public void dayOfTheWeek003() {
        ImmutualDate date = new ImmutualDate(18, 1, 1892); // * Oliver Hardy
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.MONDAY, weekday);
    }

    @Test
    public void dayOfTheWeek004() {
        ImmutualDate date = new ImmutualDate(9, 11, 1989); // Mauerfall
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.THURSDAY, weekday);
    }

    @Test
    public void dayOfTheWeek005() {
        ImmutualDate date = new ImmutualDate(24, 4, 1982); // ?
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.SATURDAY, weekday);
    }

    @Test
    public void dayOfTheWeek006() {
        ImmutualDate date = new ImmutualDate(18, 9, 1783); // ?
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.THURSDAY, weekday);
    }

    @Test
    public void dayOfTheWeek007() {
        ImmutualDate date = new ImmutualDate(1, 1, 2000); // ?
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.SATURDAY, weekday);
    }

    @Test
    public void dayOfTheWeek008() {
        ImmutualDate date = new ImmutualDate(19, 6, 2054); // ?
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.FRIDAY, weekday);
    }

    @Test
    public void dayOfTheWeek009() {
        ImmutualDate date = new ImmutualDate(5, 1, 2012); // heute
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.THURSDAY, weekday);
    }

    @Test
    public void dayOfTheWeek010() {
        ImmutualDate date = new ImmutualDate(20, 12, 2012); // 70. Geb. Gerd
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.THURSDAY, weekday);
    }

    @Test
    public void dayOfTheWeek011() {
        ImmutualDate date = new ImmutualDate(31, 3, 2021); // heute an einem anderen Tag
        Weekday weekday = date.dayOfTheWeek();

        assertEquals(Weekday.WEDNESDAY, weekday);
    }

    @Test
    public void isWorkDay001() {
        ImmutualDate date = new ImmutualDate(5, 1, 2012); // heute
        boolean isWorkDay = date.isWorkDay();
        assertTrue(isWorkDay);
    }

    @Test
    public void isWorkDay002() {
        ImmutualDate date = new ImmutualDate(6, 1, 2012); // morgen
        boolean isWorkDay = date.isWorkDay();
        assertTrue(isWorkDay);
    }

    @Test
    public void isWorkDay003() {
        ImmutualDate date = new ImmutualDate(7, 1, 2012); // übermorgen
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay004() {
        ImmutualDate date = new ImmutualDate(31, 12, 2011);
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay005() {
        ImmutualDate date = new ImmutualDate(31, 12, 2010);
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay006() {
        ImmutualDate date = new ImmutualDate(1, 1, 2013);
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay007() {
        ImmutualDate date = new ImmutualDate(17, 5, 2012); // Christi Himmelfahrt
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay008() {
        ImmutualDate date = new ImmutualDate(16, 3, 2013); // Samstag
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay009() {
        ImmutualDate date = new ImmutualDate(17, 3, 2013); // Sonntag
        boolean isWorkDay = date.isWorkDay();
        assertFalse(isWorkDay);
    }

    @Test
    public void isWorkDay010() {
        ImmutualDate date = new ImmutualDate( 6, 1, 2020); // Montag
        boolean isWorkDay = date.isWorkDay();
        assertTrue(isWorkDay);
    }

    @Test
    public void isWeekendOfWeekday() {
        assertFalse(ImmutualDate.isWeekend(Weekday.MONDAY));
        assertFalse(ImmutualDate.isWeekend(Weekday.TUESDAY));
        assertFalse(ImmutualDate.isWeekend(Weekday.WEDNESDAY));
        assertFalse(ImmutualDate.isWeekend(Weekday.THURSDAY));
        assertFalse(ImmutualDate.isWeekend(Weekday.FRIDAY));
        assertTrue(ImmutualDate.isWeekend(Weekday.SATURDAY));
        assertTrue(ImmutualDate.isWeekend(Weekday.SUNDAY));
    }

    @Test
    public void isWeekendMonday() {
        ImmutualDate date = new ImmutualDate(22, 11, 2021); // Montag
        assertFalse(date.isWeekend());
    }

    @Test
    public void isWeekendTuesday() {
        ImmutualDate date = new ImmutualDate(23, 11, 2021); // Dienstag
        assertFalse(date.isWeekend());
    }

    @Test
    public void isWeekendWednesday() {
        ImmutualDate date = new ImmutualDate(24, 11, 2021); // Mittwoch
        assertFalse(date.isWeekend());
    }

    @Test
    public void isWeekendThursday() {
        ImmutualDate date = new ImmutualDate(25, 11, 2021); // Donnerstag
        assertFalse(date.isWeekend());
    }

    @Test
    public void isWeekendFriday() {
        ImmutualDate date = new ImmutualDate(26, 11, 2021); // Freitag
        assertFalse(date.isWeekend());
    }

    @Test
    public void isWeekendSaturday() {
        ImmutualDate date = new ImmutualDate(27, 11, 2021); // Samstag
        assertTrue(date.isWeekend());
    }

    @Test
    public void isWeekendSunday() {
        ImmutualDate date = new ImmutualDate(28, 11, 2021); // Sonntag
        assertTrue(date.isWeekend());
    }

    @Test
    public void before01() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(17, 5, 2012);
        boolean expected = false;
        boolean actual   = date1.before(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void before02() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(18, 5, 2012);
        boolean expected = true;
        boolean actual   = date1.before(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void before03() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(16, 5, 2012);
        boolean expected = false;
        boolean actual   = date1.before(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void before04() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(18, 4, 2012);
        boolean expected = false;
        boolean actual   = date1.before(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void before05() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(18, 5, 2011);
        boolean expected = false;
        boolean actual   = date1.before(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void beforeOrEqual01() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(17, 5, 2012);
        boolean expected = true;
        boolean actual   = date1.beforeOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void beforeOrEqual02() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(18, 5, 2012);
        boolean expected = true;
        boolean actual   = date1.beforeOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void beforeOrEqual03() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(16, 5, 2012);
        boolean expected = false;
        boolean actual   = date1.beforeOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void beforeOrEqual04() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(18, 4, 2012);
        boolean expected = false;
        boolean actual   = date1.beforeOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void beforeOrEqual05() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(18, 5, 2011);
        boolean expected = false;
        boolean actual   = date1.beforeOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void after01() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(17, 5, 2012);
        boolean expected = false;
        boolean actual   = date1.after(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void after02() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(16, 5, 2012);
        boolean expected = true;
        boolean actual   = date1.after(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void afterOrEqual01() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(17, 5, 2012);
        boolean expected = true;
        boolean actual   = date1.afterOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void afterOrEqual02() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(16, 5, 2012);
        boolean expected = true;
        boolean actual   = date1.afterOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void afterOrEqual03() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(18, 5, 2012);
        boolean expected = false;
        boolean actual   = date1.afterOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void afterOrEqual04() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(17, 6, 2012);
        boolean expected = false;
        boolean actual   = date1.afterOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void afterOrEqual05() {
        ImmutualDate date1 = new ImmutualDate(17, 5, 2012);
        ImmutualDate date2 = new ImmutualDate(16, 5, 2013);
        boolean expected = false;
        boolean actual   = date1.afterOrEqual(date2);
        assertEquals(expected, actual);
    }

    @Test
    public void between01() {
        ImmutualDate date = new ImmutualDate(17, 5, 2012);
        ImmutualDate first = new ImmutualDate(16, 5, 2012);
        ImmutualDate last  = new ImmutualDate(18, 5, 2012);
        boolean expected = true;
        boolean actual   = date.between(first, last);
        assertEquals(expected, actual);
    }

    @Test
    public void between02() {
        ImmutualDate date = new ImmutualDate(16, 5, 2012);
        ImmutualDate first = new ImmutualDate(16, 5, 2012);
        ImmutualDate last  = new ImmutualDate(18, 5, 2012);
        boolean expected = true;
        boolean actual   = date.between(first, last);
        assertEquals(expected, actual);
    }

    @Test
    public void between03() {
        ImmutualDate date = new ImmutualDate(16, 5, 2012);
        ImmutualDate first = new ImmutualDate(16, 5, 2012);
        ImmutualDate last  = new ImmutualDate(16, 5, 2012);
        boolean expected = true;
        boolean actual   = date.between(first, last);
        assertEquals(expected, actual);
    }

    @Test
    public void between04() {
        ImmutualDate date = new ImmutualDate(18, 5, 2012);
        ImmutualDate first = new ImmutualDate(19, 5, 2012);
        ImmutualDate last  = new ImmutualDate(22, 5, 2012);
        boolean expected = false;
        boolean actual   = date.between(first, last);
        assertEquals(expected, actual);
    }

    @Test
    public void calculateDayDistanceTo001() {
        ImmutualDate date1 = new ImmutualDate(6, 1, 2012); // heute
        ImmutualDate date2 = new ImmutualDate(6, 1, 2012); // heute
        int distance = date1.calculateDayDistanceTo(date2);
        int expected = 0;
        assertEquals(expected, distance);
    }

    @Test
    public void calculateDayDistanceTo002() {
        ImmutualDate date1 = new ImmutualDate(6, 1, 2012); // heute
        ImmutualDate date2 = new ImmutualDate(7, 1, 2012); // morgen
        int distance = date1.calculateDayDistanceTo(date2);
        int expected = 1;
        assertEquals(expected, distance);
    }

    @Test
    public void calculateDayDistanceTo003() {
        ImmutualDate date1 = new ImmutualDate(6, 1, 2012); // heute
        ImmutualDate date2 = new ImmutualDate(5, 1, 2012); // gestern
        int distance = date1.calculateDayDistanceTo(date2);
        int expected = -1;
        assertEquals(expected, distance);
    }

    @Test
    public void calculateDayDistanceTo004() {
        ImmutualDate date1 = new ImmutualDate(6, 1, 2012); // heute
        ImmutualDate date2 = new ImmutualDate(8, 1, 2012); // übermorgen
        int distance = date1.calculateDayDistanceTo(date2);
        int expected = 2;
        assertEquals(expected, distance);
    }

    @Test
    public void calculateDayDistanceTo005() {
        ImmutualDate date1 = new ImmutualDate(6, 1, 2012); // heute
        ImmutualDate date2 = new ImmutualDate(4, 1, 2012); // vorgestern
        int distance = date1.calculateDayDistanceTo(date2);
        int expected = -2;
        assertEquals(expected, distance);
    }

    @Test
    public void addDaysAndCalculateDistance() {
        /* Hier wird getestet, ob addDays und calculateDayDistanceTo zueinander
         * passen, indem für alle validen Datumswerte zwischen dem 1.1.1895 und
         * dem 31.12.2020 das Datum im Abstand von -400 bis +400 berechnet
         * werden und dann vergleichen wird, ob die Abstandsberechnung den
         * gleichen Abstand liefert.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                for (int day=1; day<=31; ++day) {
                    ImmutualDate date1 = new ImmutualDate(day, month, year);
                    if (date1.isValid()) {
                        for (int abstand=-400; abstand<=400; ++abstand) {
                            ImmutualDate date2 = date1.addDays(abstand);
                            int abstand2 = date1.calculateDayDistanceTo(date2);
                            assertEquals(abstand, abstand2);
                        }
                    }
                }
            }
        }
    }

    @Test
    public void addDaysAndCompareCalculatings() {
        /* Hier wird getestet, ob calculateDayDifference und
         * calculateDayDistanceTo die gleichen Ergebnisse liefern,
         * indem für alle validen Datumswerte zwischen dem 1.1.1895 und
         * dem 31.12.2020 das Datum im Abstand von -400 bis +400 berechnet
         * werden und dann vergleichen wird, ob die Abstandsberechnung den
         * gleichen Abstand liefert.
         */
        for (int year=1895; year<=2020; ++year) {
            for (int month=1; month<=12; ++month) {
                for (int day=1; day<=31; ++day) {
                    ImmutualDate date1 = new ImmutualDate(day, month, year);
                    if (date1.isValid()) {
                        for (int abstand=-400; abstand<=400; ++abstand) {
                            ImmutualDate date2 = date1.addDays(abstand);
                            int abstand2 = date1.calculateDayDistanceTo(date2);
                            int abstand3 = date1.calculateDayDifference(date2);
                            assertEquals(abstand2, abstand3);
                        }
                    }
                }
            }
        }
    }

    @Test
    public void normalise1() {
        // 00.05.2003 -> 30.04.2003
        ImmutualDate wrongDate = new ImmutualDate( 0,  5, 2003);
        ImmutualDate rightDate = new ImmutualDate(30,  4, 2003);
        wrongDate = wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void normalise2() {
        // -3.05.2003 -> 27.04.2003
        ImmutualDate wrongDate = new ImmutualDate(-3,  5, 2003);
        ImmutualDate rightDate = new ImmutualDate(27,  4, 2003);
        wrongDate = wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void normalise3() {
        // 32.05.2003 -> 01.06.2003
        ImmutualDate wrongDate = new ImmutualDate(32,  5, 2003);
        ImmutualDate rightDate = new ImmutualDate( 1,  6, 2003);
        wrongDate = wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void normalise4() {
        // 12.00.2011 -> 12.12.2010
        ImmutualDate wrongDate = new ImmutualDate(12,  0, 2011);
        ImmutualDate rightDate = new ImmutualDate(12, 12, 2010);
        wrongDate = wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void normalise5() {
        // 23.-1.2012 -> 23.11.2011
        ImmutualDate wrongDate = new ImmutualDate(23, -1, 2012);
        ImmutualDate rightDate = new ImmutualDate(23, 11, 2011);
        wrongDate = wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void normalise6() {
        // 03.13.2011 -> 03.01.2012
        ImmutualDate wrongDate = new ImmutualDate( 3, 13, 2011);
        ImmutualDate rightDate = new ImmutualDate( 3,  1, 2012);
        wrongDate = wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void normaliseLeapYear() {
        // 29.02.2012 -> 29.02.2012
        ImmutualDate wrongDate = new ImmutualDate(29,  2, 2012);
        ImmutualDate rightDate = new ImmutualDate(29,  2, 2012);
        wrongDate = wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void normaliseLeapYearAfterYearAdd() {
        // 29.02.2013 -> 03.01.2013
        ImmutualDate wrongDate = new ImmutualDate(29,  2, 2013);
        ImmutualDate rightDate = new ImmutualDate( 1,  3, 2013);
        wrongDate = wrongDate.normalise();
        assertEquals(rightDate, wrongDate);
    }

    @Test
    public void addYearLeapYearAfterYearAdd() {
        // 29.02.2012 + 1 Jahr -> 01.03.2013
        ImmutualDate startDate = new ImmutualDate(29,  2, 2012);
        ImmutualDate nextYearDate = startDate.addYears(1);
        ImmutualDate rightDate = new ImmutualDate( 1,  3, 2013);
        assertEquals(rightDate, nextYearDate);
    }

    @Test
    public void addYearLeapYearAfterMonthAdd() {
        // 29.01.2013 + 1 Monat -> 01.03.2013
        ImmutualDate startDate = new ImmutualDate(29,  1, 2013);
        ImmutualDate nextYearDate = startDate.addMonths(1);
        ImmutualDate rightDate = new ImmutualDate( 1,  3, 2013);
        assertEquals(rightDate, nextYearDate);
    }

    @Test
    public void addYearLeapYearAfterDayAdd() {
        // 28.02.2013 + 1 Tag -> 01.03.2013
        ImmutualDate startDate = new ImmutualDate(28,  2, 2013);
        ImmutualDate nextYearDate = startDate.addDays(1);
        ImmutualDate rightDate = new ImmutualDate( 1,  3, 2013);
        assertEquals(rightDate, nextYearDate);
    }

    @Test
    public void addMonthToJanuary31InLeapYear() {
        // 31.01.2012 + 1 Monat -> 62.01.2012 -> 02.03.2012
        ImmutualDate startDate  = new ImmutualDate(31,  1, 2012);
        ImmutualDate expectedDate = new ImmutualDate( 2,  3, 2012);
        ImmutualDate actualDate = startDate.addMonths(1);
        assertEquals(expectedDate, actualDate);
    }

    @Test
    public void addMonthToJanuary31InNormalYear() {
        // 31.01.2013 + 1 Monat -> 62.01.2013 -> 03.03.2013
        ImmutualDate startDate  = new ImmutualDate(31,  1, 2013);
        ImmutualDate targetDate = new ImmutualDate( 3,  3, 2013);
        startDate = startDate.addMonths(1);
        assertEquals(targetDate, startDate);
    }

    @Test
    public void addMonthToMarch31() {
        // 31.03.2012 + 1 Monat -> 62.03.2012 -> 01.05.2012
        ImmutualDate startDate  = new ImmutualDate(31,  3, 2012);
        ImmutualDate targetDate = new ImmutualDate( 1,  5, 2012);
        startDate = startDate.addMonths(1);
        assertEquals(targetDate, startDate);
    }

    /*
     * Hier könnte man überlegen, die Logik zu ändern. Ein + 1 Monat bleibt immer im Folgemonat...
     * Vgl. Clean code S. 170 unten.
     */

    @Test
    public void toStringTest1() {
        ImmutualDate date = new ImmutualDate(31, 3, 2012);
        String excpeted = "31.03.2012";
        String actual = date.toString();
        assertEquals(excpeted, actual);
    }

    @Test
    public void toStringTest2() {
        ImmutualDate date = new ImmutualDate(31, 3, 2012);
        String excpeted = "2012/03/31";
        String actual = date.toStringInternational();
        assertEquals(excpeted, actual);
    }

    @Test
    public void asYyyyMmDd() {
        ImmutualDate date = new ImmutualDate(31, 3, 2012);
        String excpeted = "20120331";
        String actual = date.asYyyyMmDd();
        assertEquals(excpeted, actual);
    }

    @Test
    public void asYyyyMinusMmMinusDd() {
        ImmutualDate date = new ImmutualDate(31, 3, 2012);
        String excpeted = "2012-03-31";
        String actual = date.asYyyyMinusMmMinusDd();
        assertEquals(excpeted, actual);
    }

    @Test
    public void asDdMmYyyy() {
        ImmutualDate date = new ImmutualDate(31, 3, 2012);
        String excpeted = "31032012";
        String actual = date.asDdMmYyyy();
        assertEquals(excpeted, actual);
    }

    @Test
    public void getMonthAsFirstThreeLettersOfEnglishNameMJan() {
        ImmutualDate date = new ImmutualDate(20, 1, 2020);
        String excpeted = "JAN";
        String actual = date.getMonthAsFirstThreeLettersOfEnglishName();
        assertEquals(excpeted, actual);
    }

    @Test
    public void getMonthAsFirstThreeLettersOfEnglishNameFeb() {
        ImmutualDate date = new ImmutualDate(11, 2, 2019);
        String excpeted = "FEB";
        String actual = date.getMonthAsFirstThreeLettersOfEnglishName();
        assertEquals(excpeted, actual);
    }

    @Test
    public void getMonthAsFirstThreeLettersOfEnglishNameMar() {
        ImmutualDate date = new ImmutualDate( 2, 3, 2012);
        String excpeted = "MAR";
        String actual = date.getMonthAsFirstThreeLettersOfEnglishName();
        assertEquals(excpeted, actual);
    }

    @Test
    public void getMonthAsFirstThreeLettersOfEnglishNameApr() {
        ImmutualDate date = new ImmutualDate( 4, 4, 2020);
        String excpeted = "APR";
        String actual = date.getMonthAsFirstThreeLettersOfEnglishName();
        assertEquals(excpeted, actual);
    }

    @Test
    public void getMonthAsFirstThreeLettersOfEnglishNameMay() {
        ImmutualDate date = new ImmutualDate( 1, 5, 2020);
        String excpeted = "MAY";
        String actual = date.getMonthAsFirstThreeLettersOfEnglishName();
        assertEquals(excpeted, actual);
    }

    @Test
    public void getMonthAsFirstThreeLettersOfEnglishNameJun() {
        ImmutualDate date = new ImmutualDate( 2, 6, 1967);
        String excpeted = "JUN";
        String actual = date.getMonthAsFirstThreeLettersOfEnglishName();
        assertEquals(excpeted, actual);
    }

    @Test
    public void getMonthAsFirstThreeLettersOfEnglishNameJul() {
        ImmutualDate date = new ImmutualDate( 6, 7, 1988);
        String excpeted = "JUL";
        String actual = date.getMonthAsFirstThreeLettersOfEnglishName();
        assertEquals(excpeted, actual);
    }

    @Test
    public void getMonthAsFirstThreeLettersOfEnglishNameAug() {
        ImmutualDate date = new ImmutualDate(21, 8, 2020);
        String excpeted = "AUG";
        String actual = date.getMonthAsFirstThreeLettersOfEnglishName();
        assertEquals(excpeted, actual);
    }

    @Test
    public void getMonthAsFirstThreeLettersOfEnglishNameSep() {
        ImmutualDate date = new ImmutualDate(11, 9, 2020);
        String excpeted = "SEP";
        String actual = date.getMonthAsFirstThreeLettersOfEnglishName();
        assertEquals(excpeted, actual);
    }

    @Test
    public void getMonthAsFirstThreeLettersOfEnglishNameOct() {
        ImmutualDate date = new ImmutualDate( 4, 10, 2020);
        String excpeted = "OCT";
        String actual = date.getMonthAsFirstThreeLettersOfEnglishName();
        assertEquals(excpeted, actual);
    }

    @Test
    public void getMonthAsFirstThreeLettersOfEnglishNameNov() {
        ImmutualDate date = new ImmutualDate( 4, 11, 2020);
        String excpeted = "NOV";
        String actual = date.getMonthAsFirstThreeLettersOfEnglishName();
        assertEquals(excpeted, actual);
    }

    @Test
    public void getMonthAsFirstThreeLettersOfEnglishNameDec() {
        ImmutualDate date = new ImmutualDate( 4, 12, 2020);
        String excpeted = "DEC";
        String actual = date.getMonthAsFirstThreeLettersOfEnglishName();
        assertEquals(excpeted, actual);
    }

    @Test
    public void toSasFormat() {
        ImmutualDate date = new ImmutualDate(31, 3, 2012);
        String excpeted = "31MAR2012";
        String actual = date.toSasFormat();
        assertEquals(excpeted, actual);
    }

    @Test
    public void startDateTest01() {
        String input = "26. 10./ 02. 11. 2000";
        ImmutualDate date = new ImmutualDate(input);
        assertEquals("00.00.0000", date.toString());
        assertEquals(false, date.isValid());
    }

    @Test
    public void comparator01() {
        List<ImmutualDate> dates = new ArrayList<>();
        dates.add(new ImmutualDate(24, 12, 2014));
        dates.add(new ImmutualDate(10, 12, 2014));
        dates.add(new ImmutualDate(31, 12, 2014));
        dates.add(new ImmutualDate(25, 12, 2014));
        dates.add(new ImmutualDate( 9, 12, 2014));
        dates.add(new ImmutualDate(26, 12, 2014));

        Collections.sort(dates);

        assertEquals( 9, dates.get(0).getDay());
        assertEquals(10, dates.get(1).getDay());
        assertEquals(24, dates.get(2).getDay());
        assertEquals(25, dates.get(3).getDay());
        assertEquals(26, dates.get(4).getDay());
        assertEquals(31, dates.get(5).getDay());
    }

    @Test
    public void comparator02() {
        List<ImmutualDate> dates = new ArrayList<>();
        dates.add(new ImmutualDate(10,  6, 2014));
        dates.add(new ImmutualDate(10,  3, 2014));
        dates.add(new ImmutualDate(10,  1, 2014));
        dates.add(new ImmutualDate(10, 10, 2014));
        dates.add(new ImmutualDate(10, 12, 2014));
        dates.add(new ImmutualDate(10, 11, 2014));

        Collections.sort(dates);

        assertEquals( 1, dates.get(0).getMonth());
        assertEquals( 3, dates.get(1).getMonth());
        assertEquals( 6, dates.get(2).getMonth());
        assertEquals(10, dates.get(3).getMonth());
        assertEquals(11, dates.get(4).getMonth());
        assertEquals(12, dates.get(5).getMonth());
    }

    @Test
    public void comparator03() {
        List<ImmutualDate> dates = new ArrayList<>();
        dates.add(new ImmutualDate(29,  6, 2014));
        dates.add(new ImmutualDate(30,  3, 2014));
        dates.add(new ImmutualDate(31,  1, 2014));
        dates.add(new ImmutualDate(28, 10, 2014));
        dates.add(new ImmutualDate(26, 12, 2014));
        dates.add(new ImmutualDate(27, 11, 2014));

        Collections.sort(dates);

        assertEquals( 1, dates.get(0).getMonth());
        assertEquals( 3, dates.get(1).getMonth());
        assertEquals( 6, dates.get(2).getMonth());
        assertEquals(10, dates.get(3).getMonth());
        assertEquals(11, dates.get(4).getMonth());
        assertEquals(12, dates.get(5).getMonth());
    }

    @Test
    public void comparator04() {
        List<ImmutualDate> dates = new ArrayList<>();
        dates.add(new ImmutualDate(29,  6, 2012));
        dates.add(new ImmutualDate(30,  3, 2013));
        dates.add(new ImmutualDate(31,  1, 2014));
        dates.add(new ImmutualDate(28, 10, 2011));
        dates.add(new ImmutualDate(26, 12, 2009));
        dates.add(new ImmutualDate(27, 11, 2010));

        Collections.sort(dates);

        assertEquals(2009, dates.get(0).getYear());
        assertEquals(2010, dates.get(1).getYear());
        assertEquals(2011, dates.get(2).getYear());
        assertEquals(2012, dates.get(3).getYear());
        assertEquals(2013, dates.get(4).getYear());
        assertEquals(2014, dates.get(5).getYear());
    }

    @Test
    public void equalsTest() {
        ImmutualDate date1  = new ImmutualDate(30, 12, 2016);
        ImmutualDate date2  = new ImmutualDate(30, 12, 2016);
        assertEquals(date1,  date2);
    }

    @Test
    public void generateNextWorkDayNextDay() {
        ImmutualDate date = new ImmutualDate(14,  4, 2021);
        ImmutualDate nextWorkDay = date.generateNextWorkDay();
        assertEquals(nextWorkDay,  new ImmutualDate(15,  4, 2021));
    }

    @Test
    public void generateNextWorkDayFridayToMonday() {
        ImmutualDate date = new ImmutualDate(16,  4, 2021);
        ImmutualDate nextWorkDay = date.generateNextWorkDay();
        assertEquals(nextWorkDay,  new ImmutualDate(19,  4, 2021));
    }

    @Test
    public void generateNextWorkDayGreenThursdayToEasterTuesday() {
        ImmutualDate date = new ImmutualDate(1,  4, 2021);
        ImmutualDate nextWorkDay = date.generateNextWorkDay();
        assertEquals(nextWorkDay,  new ImmutualDate(6,  4, 2021));
    }

    @Test
    public void toXml() {
        ImmutualDate date = new ImmutualDate(26, 11, 2021);
        String actual = date.toXml();
        String expected = "<immutual-date>26.11.2021</immutual-date>";
        assertEquals(expected,  actual);
    }

    @Test
    public void createFromXml() {
        String line = "<immutual-date>26.11.2021</immutual-date>";
        ImmutualDate actual = ImmutualDate.createFromXml(line);
        ImmutualDate expected = new ImmutualDate(26, 11, 2021);
        assertEquals(expected,  actual);
    }

    @Test
    public void getWeekdayBeforeMonday() {
        ImmutualDate date = new ImmutualDate( 1, 12, 2021); // Mittwoch
        Weekday weekday = Weekday.MONDAY;
        ImmutualDate actual = date.getWeekdayBefore(weekday);
        ImmutualDate expected = new ImmutualDate(29, 11, 2021);
        assertEquals(expected,  actual);
    }

    @Test
    public void getWeekdayBeforeTuesday() {
        ImmutualDate date = new ImmutualDate( 1, 12, 2021); // Mittwoch
        Weekday weekday = Weekday.TUESDAY;
        ImmutualDate actual = date.getWeekdayBefore(weekday);
        ImmutualDate expected = new ImmutualDate(30, 11, 2021);
        assertEquals(expected,  actual);
    }

    @Test
    public void getWeekdayBeforeWednesday() {
        ImmutualDate date = new ImmutualDate( 1, 12, 2021); // Mittwoch
        Weekday weekday = Weekday.WEDNESDAY;
        ImmutualDate actual = date.getWeekdayBefore(weekday);
        ImmutualDate expected = new ImmutualDate(24, 11, 2021);
        assertEquals(expected,  actual);
    }

    @Test
    public void getWeekdayBeforeThursday() {
        ImmutualDate date = new ImmutualDate( 1, 12, 2021); // Mittwoch
        Weekday weekday = Weekday.THURSDAY;
        ImmutualDate actual = date.getWeekdayBefore(weekday);
        ImmutualDate expected = new ImmutualDate(25, 11, 2021);
        assertEquals(expected,  actual);
    }

    @Test
    public void getWeekdayBeforeFriday() {
        ImmutualDate date = new ImmutualDate( 1, 12, 2021); // Mittwoch
        Weekday weekday = Weekday.FRIDAY;
        ImmutualDate actual = date.getWeekdayBefore(weekday);
        ImmutualDate expected = new ImmutualDate(26, 11, 2021);
        assertEquals(expected,  actual);
    }

    @Test
    public void getWeekdayBeforeSaturday() {
        ImmutualDate date = new ImmutualDate( 1, 12, 2021); // Mittwoch
        Weekday weekday = Weekday.SATURDAY;
        ImmutualDate actual = date.getWeekdayBefore(weekday);
        ImmutualDate expected = new ImmutualDate(27, 11, 2021);
        assertEquals(expected,  actual);
    }

    @Test
    public void getWeekdayBeforeSunday() {
        ImmutualDate date = new ImmutualDate( 1, 12, 2021); // Mittwoch
        Weekday weekday = Weekday.SUNDAY;
        ImmutualDate actual = date.getWeekdayBefore(weekday);
        ImmutualDate expected = new ImmutualDate(28, 11, 2021);
        assertEquals(expected,  actual);
    }

    @Test
    public void getMonthAsGermanWordJan() {
        ImmutualDate date = new ImmutualDate( 1,  1, 2022);
        String actual = date.getMonthAsGermanWord();
        String expected = "Januar";
        assertEquals(expected, actual);
    }

    @Test
    public void getMonthAsGermanWordFeb() {
        ImmutualDate date = new ImmutualDate( 1,  2, 2022);
        String actual = date.getMonthAsGermanWord();
        String expected = "Februar";
        assertEquals(expected, actual);
    }

    @Test
    public void getMonthAsGermanWordMar() {
        ImmutualDate date = new ImmutualDate( 1,  3, 2022);
        String actual = date.getMonthAsGermanWord();
        String expected = "März";
        assertEquals(expected, actual);
    }

    @Test
    public void getMonthAsGermanWordApr() {
        ImmutualDate date = new ImmutualDate( 1,  4, 2022);
        String actual = date.getMonthAsGermanWord();
        String expected = "April";
        assertEquals(expected, actual);
    }

    @Test
    public void getMonthAsGermanWordMai() {
        ImmutualDate date = new ImmutualDate( 1,  5, 2022);
        String actual = date.getMonthAsGermanWord();
        String expected = "Mai";
        assertEquals(expected, actual);
    }

    @Test
    public void getMonthAsGermanWordJun() {
        ImmutualDate date = new ImmutualDate( 1,  6, 2022);
        String actual = date.getMonthAsGermanWord();
        String expected = "Juni";
        assertEquals(expected, actual);
    }

    @Test
    public void getMonthAsGermanWordJul() {
        ImmutualDate date = new ImmutualDate( 1,  7, 2022);
        String actual = date.getMonthAsGermanWord();
        String expected = "Juli";
        assertEquals(expected, actual);
    }

    @Test
    public void getMonthAsGermanWordAug() {
        ImmutualDate date = new ImmutualDate( 1,  8, 2022);
        String actual = date.getMonthAsGermanWord();
        String expected = "August";
        assertEquals(expected, actual);
    }

    @Test
    public void getMonthAsGermanWordSep() {
        ImmutualDate date = new ImmutualDate( 1,  9, 2022);
        String actual = date.getMonthAsGermanWord();
        String expected = "September";
        assertEquals(expected, actual);
    }

    @Test
    public void getMonthAsGermanWordOct() {
        ImmutualDate date = new ImmutualDate( 1, 10, 2022);
        String actual = date.getMonthAsGermanWord();
        String expected = "Oktober";
        assertEquals(expected, actual);
    }

    @Test
    public void getMonthAsGermanWordNov() {
        ImmutualDate date = new ImmutualDate( 1, 11, 2022);
        String actual = date.getMonthAsGermanWord();
        String expected = "November";
        assertEquals(expected, actual);
    }

    @Test
    public void getMonthAsGermanWordDec() {
        ImmutualDate date = new ImmutualDate( 1, 12, 2022);
        String actual = date.getMonthAsGermanWord();
        String expected = "Dezember";
        assertEquals(expected, actual);
    }

}
