package de.duehl.basics.datetime.date.parse;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.datetime.date.ImmutualDate;

public class DateParserTest {

    @Test
    public void parseNormalDates() {
        List<String> datesToParse = CollectionsHelper.buildListFrom(
                "01.02.2018",
                "01. 02.    2018",
                "01.2.2018",
                "1.2.2018");
        for (String dateToParse : datesToParse) {
            DateParser parser = new DateParser(dateToParse);
            ImmutualDate date = parser.parse();
            assertEquals(1, date.getDay());
            assertEquals(2, date.getMonth());
            assertEquals(2018, date.getYear());

            assertTrue(date.isValid());
        }
    }

    @Test
    public void parseUnixDates() {
        List<String> datesToParse = CollectionsHelper.buildListFrom(
                "Wed Jan 18 12:01:50 CET 2012");
        for (String dateToParse : datesToParse) {
            DateParser parser = new DateParser(dateToParse);
            ImmutualDate date = parser.parse();
            assertEquals(18, date.getDay());
            assertEquals(1, date.getMonth());
            assertEquals(2012, date.getYear());

            assertTrue(date.isValid());
        }
    }

    @Test
    public void parseTwoDigitsDates() {
        List<String> datesToParse = CollectionsHelper.buildListFrom(
                "01.02.18",
                "01. 02.    18",
                "01.2.18",
                "1.2.18");
        for (String dateToParse : datesToParse) {
            DateParser parser = new DateParser(dateToParse);
            ImmutualDate date = parser.parse();
            assertEquals(1, date.getDay());
            assertEquals(2, date.getMonth());
            assertEquals(2018, date.getYear());

            assertTrue(date.isValid());
        }
    }

    @Test
    public void parseYYYYMMDDDates() {
        List<String> datesToParse = CollectionsHelper.buildListFrom(
                "20130822");
        for (String dateToParse : datesToParse) {
            DateParser parser = new DateParser(dateToParse);
            ImmutualDate date = parser.parse();
            assertEquals(22, date.getDay());
            assertEquals(8, date.getMonth());
            assertEquals(2013, date.getYear());

            assertTrue(date.isValid());
        }
    }

    @Test
    public void parseSlashDates() {
        List<String> datesToParse = CollectionsHelper.buildListFrom(
                "2013/08/02",
                "2013/8/02",
                "2013/8/2");
        for (String dateToParse : datesToParse) {
            DateParser parser = new DateParser(dateToParse);
            ImmutualDate date = parser.parse();
            assertEquals(2, date.getDay());
            assertEquals(8, date.getMonth());
            assertEquals(2013, date.getYear());

            assertTrue(date.isValid());
        }
    }

    @Test
    public void parseMinusDates() {
        List<String> datesToParse = CollectionsHelper.buildListFrom(
                "2013-08-02",
                "2013-8-02",
                "2013-8-2");
        for (String dateToParse : datesToParse) {
            DateParser parser = new DateParser(dateToParse);
            ImmutualDate date = parser.parse();
            assertEquals(2, date.getDay());
            assertEquals(8, date.getMonth());
            assertEquals(2013, date.getYear());

            assertTrue(date.isValid());
        }
    }

    @Test
    public void parseMonthWordsBlanksDates() {
        List<String> datesToParse = CollectionsHelper.buildListFrom(
                "1. Feb. 2012",
                "1. Feb 2012",
                "01. Feb. 2012",
                "01. Feb 2012",
                "1. Februar. 2012",
                "1. Februar 2012",
                "01. Februar. 2012",
                "01. Februar 2012");
        for (String dateToParse : datesToParse) {
            DateParser parser = new DateParser(dateToParse);
            ImmutualDate date = parser.parse();
            assertEquals(1, date.getDay());
            assertEquals(2, date.getMonth());
            assertEquals(2012, date.getYear());

            assertTrue(date.isValid());
        }
    }

    @Test
    public void parseMonthWordsBlanksShortYearDates() {
        List<String> datesToParse = CollectionsHelper.buildListFrom(
                "1. Feb. 12",
                "1. Feb 12",
                "01. Feb. 12",
                "01. Feb 12",
                "1. Februar. 12",
                "1. Februar 12",
                "01. Februar. 12",
                "01. Februar 12");
        for (String dateToParse : datesToParse) {
            DateParser parser = new DateParser(dateToParse);
            ImmutualDate date = parser.parse();
            assertEquals(1, date.getDay());
            assertEquals(2, date.getMonth());
            assertEquals(2012, date.getYear());

            assertTrue(date.isValid());
        }
    }

    @Test
    public void parseMonthWordsHyphensDates() {
        List<String> datesToParse = CollectionsHelper.buildListFrom(
                "1.-Feb.-2012",
                "1.-Feb-2012",
                "01.-Feb.-2012",
                "01.-Feb-2012",
                "1.-Februar.-2012",
                "1.-Februar-2012",
                "01.-Februar.-2012",
                "01.-Februar-2012",
                "1-Feb.-2012",
                "1-Feb-2012",
                "01-Feb.-2012",
                "01-Feb-2012",
                "1-Februar.-2012",
                "1-Februar-2012",
                "01-Februar.-2012",
                "01-Februar-2012");
        for (String dateToParse : datesToParse) {
            DateParser parser = new DateParser(dateToParse);
            ImmutualDate date = parser.parse();
            assertEquals(1, date.getDay());
            assertEquals(2, date.getMonth());
            assertEquals(2012, date.getYear());

            assertTrue(date.isValid());
        }
    }

    @Test
    public void parseMonthWordsHyphensShortYearDates() {
        List<String> datesToParse = CollectionsHelper.buildListFrom(
                "1.-Feb.-12",
                "1.-Feb-12",
                "01.-Feb.-12",
                "01.-Feb-12",
                "1.-Februar.-12",
                "1.-Februar-12",
                "01.-Februar.-12",
                "01.-Februar-12",
                "1-Feb.-12",
                "1-Feb-12",
                "01-Feb.-12",
                "01-Feb-12",
                "1-Februar.-12",
                "1-Februar-12",
                "01-Februar.-12",
                "01-Februar-12");
        for (String dateToParse : datesToParse) {
            DateParser parser = new DateParser(dateToParse);
            ImmutualDate date = parser.parse();
            assertEquals(1, date.getDay());
            assertEquals(2, date.getMonth());
            assertEquals(2012, date.getYear());

            assertTrue(date.isValid());
        }
    }

}
