package de.duehl.basics.text;

/*
 * Copyright 2016 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import org.junit.Test;

public class TextPositionTest {

    @Test (expected = IllegalArgumentException.class)
    public void createWithSmallerEnd() {
        new TextPosition(5, 4);
    }

    @Test (expected = IllegalArgumentException.class)
    public void createWithEndNegativ() {
        new TextPosition(-5, -2);
    }

    @Test (expected = IllegalArgumentException.class)
    public void createWithBeginNegativ() {
        new TextPosition(-1, 2);
    }

    @Test
    public void create() {
        TextPosition position = new TextPosition(1, 2);
        assertFalse(null == position);
    }

    @Test
    public void compareToItself() {
        TextPosition position = new TextPosition(0, 5);
        assertEquals(0, position.compareTo(position));
    }

    @Test
    public void compareToObjektWithSameValues() {
        TextPosition position1 = new TextPosition(0, 5);
        TextPosition position2 = new TextPosition(0, 5);
        assertEquals(0, position1.compareTo(position2));
        assertEquals(0, position2.compareTo(position1));
    }

    @Test
    public void compareToObjektWithSameBeginAndBiggerEnd() {
        TextPosition position1 = new TextPosition(0, 5);
        TextPosition position2 = new TextPosition(0, 7);
        assertTrue(position1.compareTo(position2) < 0);
        assertTrue(position2.compareTo(position1) > 0);
    }

    @Test
    public void compareToObjektWithSameBeginAndSmallerEnd() {
        TextPosition position1 = new TextPosition(0, 5);
        TextPosition position2 = new TextPosition(0, 3);
        assertTrue(position1.compareTo(position2) > 0);
        assertTrue(position2.compareTo(position1) < 0);
    }

    @Test
    public void compareToObjektWithSameEndAndBiggerBegin() {
        TextPosition position1 = new TextPosition(0, 5);
        TextPosition position2 = new TextPosition(2, 5);
        assertTrue(position1.compareTo(position2) < 0);
        assertTrue(position2.compareTo(position1) > 0);
    }

    @Test
    public void compareToObjektWithSameEndAndSmallerBegin() {
        TextPosition position1 = new TextPosition(1, 5);
        TextPosition position2 = new TextPosition(0, 5);
        assertTrue(position1.compareTo(position2) > 0);
        assertTrue(position2.compareTo(position1) < 0);
    }

    @Test
    public void compareToObjektWithDifferentBeginAndEndButOverlapping() {
        TextPosition position1 = new TextPosition(0, 5);
        TextPosition position2 = new TextPosition(3, 9);
        assertTrue(position1.compareTo(position2) < 0);
        assertTrue(position2.compareTo(position1) > 0);
    }

    @Test
    public void compareToObjektWithDifferentBeginAndEndNotOverlapping() {
        TextPosition position1 = new TextPosition(0, 5);
        TextPosition position2 = new TextPosition(7, 9);
        assertTrue(position1.compareTo(position2) < 0);
        assertTrue(position2.compareTo(position1) > 0);
    }

    @Test
    public void getTextPartAtPostionWithTooSmallText() {
        String fullText = "bla";
        TextPosition position = new TextPosition(5, 9);
        String actual = position.getTextPartAtPostion(fullText);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void getTextPartAtPostionWithEmptyText() {
        String fullText = "";
        TextPosition position = new TextPosition(0, 10);
        String actual = position.getTextPartAtPostion(fullText);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void getTextPartAtPostionWithEmptyTextAndEmptyPosition() {
        String fullText = "";
        TextPosition position = new TextPosition(0, 0);
        String actual = position.getTextPartAtPostion(fullText);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void getTextPartAtPostionWithNotEmptyTextAndEmptyPosition() {
        String fullText = "abcdef";
        TextPosition position = new TextPosition(1, 1);
        String actual = position.getTextPartAtPostion(fullText);
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void getTextPartAtPostionStandardCase() {
        String fullText = "abcdef";
        TextPosition position = new TextPosition(1, 3);
        String actual = position.getTextPartAtPostion(fullText);
        String expected = "bc";
        assertEquals(expected, actual);
    }

    @Test
    public void overlappStandardCaseNotOverlapping() {
        TextPosition position1 = new TextPosition(0, 3);
        TextPosition position2 = new TextPosition(5, 9);
        assertFalse(position1.overlapp(position2));
        assertFalse(position2.overlapp(position1));
    }

    @Test
    public void overlappToItself() {
        TextPosition position = new TextPosition(0, 3);
        assertTrue(position.overlapp(position));
    }

    @Test
    public void overlappWithSameValues() {
        TextPosition position1 = new TextPosition(0, 3);
        TextPosition position2 = new TextPosition(0, 3);
        assertTrue(position1.overlapp(position2));
        assertTrue(position2.overlapp(position1));
    }

    @Test
    public void overlappStandardCaseOverlapping() {
        TextPosition position1 = new TextPosition(0, 5);
        TextPosition position2 = new TextPosition(3, 9);
        assertTrue(position1.overlapp(position2));
        assertTrue(position2.overlapp(position1));
    }

    @Test
    public void overlappNotOverlappingWithoutSpaceBetween() {
        TextPosition position1 = new TextPosition(0, 3);
        TextPosition position2 = new TextPosition(3, 5);
        assertFalse(position1.overlapp(position2));
        assertFalse(position2.overlapp(position1));
    }

    @Test
    public void overlappWithInnerPosition() {
        TextPosition position1 = new TextPosition(0, 10);
        TextPosition position2 = new TextPosition(3, 5);
        assertTrue(position1.overlapp(position2));
        assertTrue(position2.overlapp(position1));
    }

    @Test
    public void meltWithItself() {
        TextPosition position = new TextPosition(0, 3);
        TextPosition meltedPosition = position.meltWith(position);
        assertEquals(position.getBegin(), meltedPosition.getBegin());
        assertEquals(position.getEnd(), meltedPosition.getEnd());
    }

    @Test
    public void meltWithSameValues() {
        TextPosition position1 = new TextPosition(0, 3);
        TextPosition position2 = new TextPosition(0, 3);
        TextPosition meltedPosition = position1.meltWith(position2);
        assertEquals(position1.getBegin(), meltedPosition.getBegin());
        assertEquals(position1.getEnd(), meltedPosition.getEnd());
        assertEquals(position2.getBegin(), meltedPosition.getBegin());
        assertEquals(position2.getEnd(), meltedPosition.getEnd());
    }

}
