package de.duehl.basics.text.html;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import org.junit.Test;

import static de.duehl.basics.text.Text.LINE_BREAK;

public class Text2HtmlBasicsTest {

    private static final String COLOR_START1       = "<span style=\"color:";
    private static final String COLOR_START2       = ";font-weight:bold\">";
    private static final String GOOD_COLOR_START   = COLOR_START1 + "008800" + COLOR_START2;
    private static final String BAD_COLOR_START    = COLOR_START1 + "FF0000" + COLOR_START2;
    private static final String MEDIUM_COLOR_START = COLOR_START1 + "FFA500" + COLOR_START2;
    private static final String EDITOR_HTML_COLOR  = "9400d3";
    private static final String EDITOR_COLOR_START = COLOR_START1 + EDITOR_HTML_COLOR + COLOR_START2;
    private static final String COLOR_OFF          = "</span>";

    @Test
    public void nestedSpans01() {
        String input = EDITOR_COLOR_START + "foo" + COLOR_OFF;
        String actual = Text2HtmlBasics.removeNestedSpans(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void nestedSpans02() {
        String coloredFoo = EDITOR_COLOR_START + "foo" + COLOR_OFF;
        String input = coloredFoo + " bar " + coloredFoo + " baz " + coloredFoo;
        String actual = Text2HtmlBasics.removeNestedSpans(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void nestedSpans03() {
        String coloredBielefeld = EDITOR_COLOR_START + "Bielefeld" + COLOR_OFF;
        String coloredHRB = EDITOR_COLOR_START + "HRB" + COLOR_OFF;
        String colored33488 = EDITOR_COLOR_START + "33488" + COLOR_OFF;
        String input = "Amtsgericht " + coloredBielefeld + " " + coloredHRB + " " + colored33488;
        String actual = Text2HtmlBasics.removeNestedSpans(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void nestedSpansRemovesCompletelyInnerPart() {
        String start = EDITOR_COLOR_START;
        String end = COLOR_OFF;
        String input = "nur " + start + "so " + start + "ein " + end + "Text" + end + " ...";
        String actual = Text2HtmlBasics.removeNestedSpans(input);
        String expected = "nur " + start + "so ein Text" + end + " ...";
        assertEquals(expected, actual);
    }

    @Test
    public void nestedSpansRemovesNested() {
        String start = EDITOR_COLOR_START;
        String end = COLOR_OFF;
        String input = "nur " + start + "so " + start + "ein " + end + "Text" + end + " ...";
        String actual = Text2HtmlBasics.removeNestedSpans(input);
        String expected = "nur " + start + "so ein Text" + end + " ...";
        assertEquals(expected, actual);
        // identisch zu nestedSpansRemovesCompletelyInnerPart, lässt sich an dieser Stelle nicht
        // sagen, ob ganz im Inneren oder überlappend.
    }

    @Test
    public void nestedSpansConnected() {
        String start = EDITOR_COLOR_START;
        String end = COLOR_OFF;
        String input = "nur " + start + "so " + end + start + "ein Text" + end + " ...";
        String actual = Text2HtmlBasics.removeNestedSpans(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void nestedSpansConnectedWithSpace() {
        String start = EDITOR_COLOR_START;
        String end = COLOR_OFF;
        String input = "nur " + start + "so " + end + " " + start + "ein Text" + end + " ...";
        String actual = Text2HtmlBasics.removeNestedSpans(input);
        String expected = input;
        assertEquals(expected, actual);
    }

    @Test
    public void nestedSpansWithLineBreak() {
        String input = GOOD_COLOR_START + BAD_COLOR_START
                + "company(" + MEDIUM_COLOR_START
                + "companyname(moBiel GmbH, 'companyname00002')" + COLOR_OFF
                + ", location(Postfach 21 90 46, 33697 Bielefeld" + LINE_BREAK
                + "Hausanschrift, null, 'location00300'), null, null, null, null, null, null, "
                + "null, 'company00001')" + COLOR_OFF + COLOR_OFF;
        String actual = Text2HtmlBasics.removeNestedSpans(input);
        String expected = GOOD_COLOR_START
                + "company("
                + "companyname(moBiel GmbH, 'companyname00002')"
                + ", location(Postfach 21 90 46, 33697 Bielefeld" + LINE_BREAK
                + "Hausanschrift, null, 'location00300'), null, null, null, null, null, null, "
                + "null, 'company00001')" + COLOR_OFF;
        assertEquals(expected, actual);
    }

}
