package de.duehl.basics.text.html;

/*
 * Copyright 2018 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;

public class UrlHelperTest {

    @Test
    public void addHttpToUtlIfMissingAdding() {
        String url = "www.example.org";
        String actual = UrlHelper.addHttpToUrlIfMissing(url);
        String expected = "http://" + url;
        assertEquals(expected, actual);
    }

    @Test
    public void addHttpToUtlIfMissingNotAddingFront() {
        String url = "http://www.example.org";
        String actual = UrlHelper.addHttpToUrlIfMissing(url);
        String expected = url;
        assertEquals(expected, actual);
    }

    @Test
    public void addHttpToUtlIfMissingNotAddingFrontHttps() {
        String url = "https://www.example.org";
        String actual = UrlHelper.addHttpToUrlIfMissing(url);
        String expected = url;
        assertEquals(expected, actual);
    }

    @Test
    public void addHttpToUtlIfMissingNotAddingAnywhere() {
        String url = "www.http.example.org";
        String actual = UrlHelper.addHttpToUrlIfMissing(url);
        String expected = url;
        assertEquals(expected, actual);
    }

    @Test
    public void concatUrlAndAdditionalNoSlash() {
        String url = "http://www.examle.org";
        String additional = "Impresum.html";
        String expected = "http://www.examle.org/Impresum.html";
        String actual = UrlHelper.concatUrlAndAdditional(url, additional);
        assertEquals(expected, actual);
    }

    @Test
    public void concatUrlAndAdditionalSlashBehindUrl() {
        String url = "http://www.examle.org/";
        String additional = "Impresum.html";
        String expected = "http://www.examle.org/Impresum.html";
        String actual = UrlHelper.concatUrlAndAdditional(url, additional);
        assertEquals(expected, actual);
    }

    @Test
    public void concatUrlAndAdditionalSlashBeforeAdditional() {
        String url = "http://www.examle.org";
        String additional = "/Impresum.html";
        String expected = "http://www.examle.org/Impresum.html";
        String actual = UrlHelper.concatUrlAndAdditional(url, additional);
        assertEquals(expected, actual);
    }

    @Test
    public void concatUrlAndAdditionalBothWithSlashes() {
        String url = "http://www.examle.org/";
        String additional = "/Impresum.html";
        String expected = "http://www.examle.org/Impresum.html";
        String actual = UrlHelper.concatUrlAndAdditional(url, additional);
        assertEquals(expected, actual);
    }

    @Test
    public void isLinkInLLinklistWithExactSameElementInList() {
        String url = "http://www.example.org/";
        List<String> urls = CollectionsHelper.buildListFrom("http://www.example.org/");
        assertTrue(UrlHelper.isLinkInLLinklist(url, urls));
    }

    @Test
    public void isLinkInLLinklistWithBigElementInList() {
        String url = "http://www.example.org/";
        List<String> urls = CollectionsHelper.buildListFrom("HTTP://WWW.EXAMPLE.ORG/");
        assertTrue(UrlHelper.isLinkInLLinklist(url, urls));
    }

    @Test
    public void isLinkInLEmptyLinklist() {
        String url = "http://www.example.org/";
        List<String> urls = CollectionsHelper.buildListFrom();
        assertFalse(UrlHelper.isLinkInLLinklist(url, urls));
    }

    @Test
    public void isLinkInLLinklistWithDifferentParameter() {
        String url = "http://www.tanzschule-darf-ich-bitten.de/?home:impressum";
        List<String> urls = CollectionsHelper
                .buildListFrom("http://www.tanzschule-darf-ich-bitten.de/?Home:Impressum");
        assertFalse(UrlHelper.isLinkInLLinklist(url, urls));
    }

    @Test
    public void areLinksEqualAtExactEqualStringsWithoutParameters() {
        String url1 = "http://www.example.org/";
        String url2 = "http://www.example.org/";
        assertTrue(UrlHelper.areLinksEqual(url1, url2));
    }

    @Test
    public void areLinksEqualAtEqualStringsWithoutParameters() {
        String url1 = "http://www.example.org/";
        String url2 = "HTTP://WWW.EXAMPLE.ORG/";
        assertTrue(UrlHelper.areLinksEqual(url1, url2));
    }

    @Test
    public void areLinksEqualAtDifferentEqualStringsWithoutParameters() {
        String url1 = "http://www.examle.org/";
        String url2 = "http://www.examle.org/hast-du-nicht-gesehen/";
        assertFalse(UrlHelper.areLinksEqual(url1, url2));
    }

    @Test
    public void areLinksEqualAtDifferentEqualStringsWithoutParameters2() {
        String url1 = "http://www.examle.org/";
        String url2 = "http://www.duehl.de/";
        assertFalse(UrlHelper.areLinksEqual(url1, url2));
    }

    @Test
    public void areLinksEqualAtDifferentEqualStringsWithoutParameters3() {
        String url1 = "http://www.examle.org/";
        String url2 = "http://www.examle.org";
        assertFalse(UrlHelper.areLinksEqual(url1, url2));
    }

    @Test
    public void areLinksEqualAtDifferentEqualStringsWithoutParameters4() {
        String url1 = "http://www.examle.org/";
        String url2 = "https://www.examle.org/";
        assertFalse(UrlHelper.areLinksEqual(url1, url2));
    }

    @Test
    public void areLinksEqualAtDifferentStringsInTheParameters() {
        String url1 = "http://www.tanzschule-darf-ich-bitten.de/?home:impressum";
        String url2 = "http://www.tanzschule-darf-ich-bitten.de/?Home:Impressum";
        assertFalse(UrlHelper.areLinksEqual(url1, url2));
    }

    @Test
    public void safelyLowerUrl1() {
        String url = "http://www.tanzschule-darf-ich-bitten.de/?Home:Impressum";
        String expected = "http://www.tanzschule-darf-ich-bitten.de/?Home:Impressum";
        String actual = UrlHelper.safelyLowerUrl(url);
        assertEquals(expected, actual);
    }

    @Test
    public void safelyLowerUrl2() {
        String url = "HTTP://WWW.TANZSCHULE-DARF-ICH-BITTEN.DE/?Home:Impressum";
        String expected = "http://www.tanzschule-darf-ich-bitten.de/?Home:Impressum";
        String actual = UrlHelper.safelyLowerUrl(url);
        assertEquals(expected, actual);
    }

    @Test
    public void safelyLowerUrl3() {
        String url = "HTTP://WWW.EXAMPLE.ORG/";
        String expected = "http://www.example.org/";
        String actual = UrlHelper.safelyLowerUrl(url);
        assertEquals(expected, actual);
    }

    @Test
    public void normalizeUrlNormalUrl() {
        String url = "http://www.example.org/";
        String expected = "http://www.example.org/";
        String actual = UrlHelper.normalizeUrl(url);
        assertEquals(expected, actual);
    }

    @Test
    public void normalizeUrlOnePoint() {
        String url = "http://www.example.org/./";
        String expected = "http://www.example.org/";
        String actual = UrlHelper.normalizeUrl(url);
        assertEquals(expected, actual);
    }

    @Test
    public void normalizeUrlOneDoublePoint() {
        String url = "http://www.example.org/../";
        String expected = "http://www.example.org/";
        String actual = UrlHelper.normalizeUrl(url);
        assertEquals(expected, actual);
    }

    @Test
    public void normalizeUrlMulti() {
        String url = "http://www.example.org/../foo/./.././../../bar/./baz";
        String expected = "http://www.example.org/foo/bar/baz";
        String actual = UrlHelper.normalizeUrl(url);
        assertEquals(expected, actual);
    }

    @Test
    public void urlEndsWithHtmlLowHtml() {
        String url = "http://www.example.org/info.html";
        assertTrue(UrlHelper.urlEndsWithHtml(url));
    }

    @Test
    public void urlEndsWithHtmlBigHtml() {
        String url = "http://www.example.org/schau/mal/hier/info.HTML";
        assertTrue(UrlHelper.urlEndsWithHtml(url));
    }

    @Test
    public void urlEndsWithHtmlMixedHtml() {
        String url = "http://www.example.org/schau/mal/hier/info.HtMl";
        assertTrue(UrlHelper.urlEndsWithHtml(url));
    }

    @Test
    public void urlEndsWithHtmlNotEndingWithHtml() {
        String url = "http://www.example.org/";
        assertFalse(UrlHelper.urlEndsWithHtml(url));
    }

    @Test
    public void urlEndsWithHtmlNotEndingWithHtmlWithSomethingBehind() {
        String url = "http://www.example.org/info.html?id=27";
        assertFalse(UrlHelper.urlEndsWithHtml(url));
    }

    @Test
    public void urlEndsWithHtmlOrPhpLowHtml() {
        String url = "http://www.example.org/info.html";
        assertTrue(UrlHelper.urlEndsWithHtmlOrPhp(url));
    }

    @Test
    public void urlEndsWithHtmlOrPhpBigHtml() {
        String url = "http://www.example.org/schau/mal/hier/info.HTML";
        assertTrue(UrlHelper.urlEndsWithHtmlOrPhp(url));
    }

    @Test
    public void urlEndsWithHtmlOrPhpMixedHtml() {
        String url = "http://www.example.org/schau/mal/hier/info.HtMl";
        assertTrue(UrlHelper.urlEndsWithHtmlOrPhp(url));
    }

    @Test
    public void urlEndsWithHtmlOrPhpNotEndingWithHtml() {
        String url = "http://www.example.org/";
        assertFalse(UrlHelper.urlEndsWithHtmlOrPhp(url));
    }

    @Test
    public void urlEndsWithHtmlOrPhpNotEndingWithHtmlWithSomethingBehind() {
        String url = "http://www.example.org/info.html?id=27";
        assertFalse(UrlHelper.urlEndsWithHtmlOrPhp(url));
    }

    @Test
    public void urlEndsWithHtmlOrPhpLowPhp() {
        String url = "http://www.example.org/info.php";
        assertTrue(UrlHelper.urlEndsWithHtmlOrPhp(url));
    }

    @Test
    public void urlEndsWithHtmlOrPhpBigPhp() {
        String url = "http://www.example.org/schau/mal/hier/info.PHP";
        assertTrue(UrlHelper.urlEndsWithHtmlOrPhp(url));
    }

    @Test
    public void urlEndsWithHtmlOrPhpMixedPhp() {
        String url = "http://www.example.org/schau/mal/hier/info.pHp";
        assertTrue(UrlHelper.urlEndsWithHtmlOrPhp(url));
    }


    @Test
    public void urlContainsOtherUrl() {
        String url = "http://www.example.org/info.html?id=27";
        String containedUrl = "/info.html";
        assertTrue(UrlHelper.urlContainsOtherUrl(url, containedUrl));
    }

    @Test
    public void urlContainsOtherUrlBig() {
        String url = "http://www.example.org/info.html?id=27";
        String containedUrl = "/INFO.HTML";
        assertTrue(UrlHelper.urlContainsOtherUrl(url, containedUrl));
    }

    @Test
    public void determineBaseUrl() {
        String url = "http://www.example.org/info.html";
        String actual = UrlHelper.determineBaseUrl(url);
        String expected = "http://www.example.org/";
        assertEquals(expected, actual);
    }

    @Test
    public void determineBaseUrlHtm() {
        String url = "http://www.example.org/info.htm";
        String actual = UrlHelper.determineBaseUrl(url);
        String expected = "http://www.example.org/";
        assertEquals(expected, actual);
    }

    @Test
    public void determineBaseUrlWithParameter() {
        String url = "http://www.example.org/info.html?id=27";
        String actual = UrlHelper.determineBaseUrl(url);
        String expected = "http://www.example.org/";
        assertEquals(expected, actual);
    }

    @Test
    public void determineBaseUrlWithMultipleHtmlFirstComesHtml() {
        String url = "http://www.example.org/info.html?id=27/info2.htm";
        String actual = UrlHelper.determineBaseUrl(url);
        String expected = "http://www.example.org/";
        assertEquals(expected, actual);
    }

    @Test
    public void determineBaseUrlWithMultipleHtmlFirstComesHtm() {
        String url = "http://www.example.org/info.htm?id=27/info2.html";
        String actual = UrlHelper.determineBaseUrl(url);
        String expected = "http://www.example.org/";
        assertEquals(expected, actual);
    }

    @Test
    public void determineBaseUrlHtmWithPath() {
        String url = "http://www.example.org/in/diesem/verzeichnis/info.htm";
        String actual = UrlHelper.determineBaseUrl(url);
        String expected = "http://www.example.org/in/diesem/verzeichnis/";
        assertEquals(expected, actual);
    }

    @Test
    public void determineBaseUrlHtmWithPathAndPhp() {
        String url = "http://www.example.org/in/diesem/verzeichnis/info.php";
        String actual = UrlHelper.determineBaseUrl(url);
        String expected = "http://www.example.org/in/diesem/verzeichnis/";
        assertEquals(expected, actual);
    }

    @Test
    public void determineBaseUrlMultipleMix() {
        String url = "http://www.example.org/in.html/diesem.htm/verzeichnis/info.php";
        String actual = UrlHelper.determineBaseUrl(url);
        String expected = "http://www.example.org/";
        assertEquals(expected, actual);
    }

    @Test
    public void determineBaseUrlMultipleMixPhpQuestionmark() {
        String url = "http://www.example.org/index.php?i=diesem/verzeichnis/base.htm";
        String actual = UrlHelper.determineBaseUrl(url);
        String expected = "http://www.example.org/index.php?i=diesem/verzeichnis/";
        assertEquals(expected, actual);
    }

    @Test
    public void determineBaseUrlMultipleMixPhpQuestionmarkDoNotCut() {
        String url = "http://www.example.org/index.php?i=122344";
        String actual = UrlHelper.determineBaseUrl(url);
        String expected = url;
        assertEquals(expected, actual);
    }

    @Test
    public void removeUrlStart01() {
        String input = "https://bla.example.org/";
        String actual = UrlHelper.removeUrlStart(input);
        String expected = "bla.example.org/";
        assertEquals(expected, actual);
    }

    @Test
    public void removeUrlStart02() {
        String input = "https://www.example.org/";
        String actual = UrlHelper.removeUrlStart(input);
        String expected = "example.org/";
        assertEquals(expected, actual);
    }

    @Test
    public void removeUrlStart03() {
        String input = "http://www.example.org/";
        String actual = UrlHelper.removeUrlStart(input);
        String expected = "example.org/";
        assertEquals(expected, actual);
    }

    @Test
    public void url2dirname01() {
        String input = "https://bla.example.org/";
        String actual = UrlHelper.url2dirname(input);
        String expected = "bla.example.org";
        assertEquals(expected, actual);
    }

    @Test
    public void url2dirname02() {
        String input = "https://www.example.org/";
        String actual = UrlHelper.url2dirname(input);
        String expected = "example.org";
        assertEquals(expected, actual);
    }

    @Test
    public void url2dirname03() {
        String input = "http://www.example.org/";
        String actual = UrlHelper.url2dirname(input);
        String expected = "example.org";
        assertEquals(expected, actual);
    }

    @Test
    public void url2dirname04() {
        String input = "https://dict.leo.org/deutsch-englisch/erkl%C3%A4rung.html";
        String actual = UrlHelper.url2dirname(input);
        String expected = "dict.leo.org_deutsch-englisch_erkl_C3_A4rung.html";
        assertEquals(expected, actual);
    }

    @Test
    public void areUrlsEqualEnoughForFramesEqualUrls() {
        String url1 = "http://www.example.org/in/diesem/verzeichnis/info.htm";
        String url2 = "http://www.example.org/in/diesem/verzeichnis/info.htm";
        assertTrue(UrlHelper.areUrlsEqualEnoughForFrames(url1, url2));
    }

    @Test
    public void areUrlsEqualEnoughForFrames1StartsWith2() {
        String url1 = "http://www.example.org/in/diesem/verzeichnis/info.htm";
        String url2 = "http://www.example.org/in/diesem/verzeichnis/";
        assertTrue(UrlHelper.areUrlsEqualEnoughForFrames(url1, url2));
    }

    @Test
    public void areUrlsEqualEnoughForFrames2StartsWith1() {
        String url1 = "http://www.example.org/in/diesem/verzeichnis/";
        String url2 = "http://www.example.org/in/diesem/verzeichnis/info.htm";
        assertTrue(UrlHelper.areUrlsEqualEnoughForFrames(url1, url2));
    }

    @Test
    public void areUrlsEqualEnoughForFramesExample1() {
        String url1 = "http://www.borussia-delmenhorst.de";
        String url2 = "http://borussia-delmenhorst.weebly.com";
        assertTrue(UrlHelper.areUrlsEqualEnoughForFrames(url1, url2));
    }

    @Test
    public void areUrlsEqualEnoughForFramesExample2() {
        String url1 = "http://borussia-delmenhorst.weebly.com";
        String url2 = "http://borussia-delmenhorst.weebly.com/impressum.html";
        assertTrue(UrlHelper.areUrlsEqualEnoughForFrames(url1, url2));
    }

    @Test
    public void areUrlsEqualEnoughForFramesNoticeThisBehavour() {
        String url1 = "http://example.com";
        String url2 = "http://example.com/de/blabla/fasel/impressum.html";
        String url3 = "http://example.com/en/glugg/urgs/sonstwas.html";
        assertTrue(UrlHelper.areUrlsEqualEnoughForFrames(url1, url2));
        assertTrue(UrlHelper.areUrlsEqualEnoughForFrames(url1, url3));
        assertFalse(UrlHelper.areUrlsEqualEnoughForFrames(url2, url3));
    }

    @Test
    public void areUrlsEqualEnoughForFramesIsTransitivInThisCase() {
        String url1 = "http://example.com";
        String url2 = "http://example.com/impress.htm";
        String url3 = "http://do-not-try-to-reach-me.com/en/glugg/urgs/impress.htm";
        assertTrue(UrlHelper.areUrlsEqualEnoughForFrames(url1, url2));
        assertTrue(UrlHelper.areUrlsEqualEnoughForFrames(url2, url3));
        assertTrue(UrlHelper.areUrlsEqualEnoughForFrames(url1, url3));
    }

    @Test
    public void createCompareUrl() {
        String url = "http://www.example.com/../impress.htm";
        String actual = UrlHelper.createCompareUrl(url);
        String expected = "example.com/impress.htm";
        assertEquals(expected, actual);
    }

    @Test
    public void createCompareUrlWithoutAnythingToNormalize() {
        String url = "http://www.example.com/impress.htm";
        String actual = UrlHelper.createCompareUrl(url);
        String expected = "example.com/impress.htm";
        assertEquals(expected, actual);
    }

    @Test
    public void createCompareUrlWithoutStartToRemove() {
        String url = "example.com/../impress.htm";
        String actual = UrlHelper.createCompareUrl(url);
        String expected = "example.com/impress.htm";
        assertEquals(expected, actual);
    }

    @Test
    public void createCompareUrlWithoutAnythingToNormalizeAndStartToRemove() {
        String url = "example.com/impress.htm";
        String actual = UrlHelper.createCompareUrl(url);
        String expected = "example.com/impress.htm";
        assertEquals(expected, actual);
    }

    @Test
    public void createCompareUrlEmptyString() {
        String url = "";
        String actual = UrlHelper.createCompareUrl(url);
        String expected = "";
        assertEquals(expected, actual);
    }

}
