package de.duehl.basics.text.xml;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;

public class XmlParameterExtractorTest {

    @Test
    public void extractParametersWithOneParameters() {
        List<String> xml = CollectionsHelper.buildListFrom(
                "<foo param=\"1\">",
                "    <bar>Bar1</bar>",
                "</foo>"
                );
        String tag = "foo";
        XmlParameterExtractor extractor = new XmlParameterExtractor(xml, tag);
        List<NamedXmlParameter> parameters = extractor.extractParameters();

        assertEquals(1, parameters.size());

        NamedXmlParameter param1 = parameters.get(0);
        assertEquals("param", param1.getName());
        assertEquals("1", param1.getContent());
    }

    @Test
    public void extractParametersWithOneParametersExpectingOne() {
        List<String> xml = CollectionsHelper.buildListFrom(
                "<foo param=\"1\">",
                "    <bar>Bar1</bar>",
                "</foo>"
                );
        String tag = "foo";
        XmlParameterExtractor extractor = new XmlParameterExtractor(xml, tag);
        List<NamedXmlParameter> parameters = extractor.extractParametersAndCheckNumberOfParameters(1);

        assertEquals(1, parameters.size());

        NamedXmlParameter param1 = parameters.get(0);
        assertEquals("param", param1.getName());
        assertEquals("1", param1.getContent());
    }

    @Test (expected = RuntimeException.class)
    public void extractParametersWithTwoParametersExpectingOne() {
        List<String> xml = CollectionsHelper.buildListFrom(
                "<foo param1=\"1\" param2=\"zwei\">",
                "    <bar>Bar1</bar>",
                "</foo>"
                );
        String tag = "foo";
        XmlParameterExtractor extractor = new XmlParameterExtractor(xml, tag);
        extractor.extractParametersAndCheckNumberOfParameters(1);
    }

    @Test (expected = RuntimeException.class)
    public void extractParametersWithOneParametersExpectingTwo() {
        List<String> xml = CollectionsHelper.buildListFrom(
                "<foo param=\"1\">",
                "    <bar>Bar1</bar>",
                "</foo>"
                );
        String tag = "foo";
        XmlParameterExtractor extractor = new XmlParameterExtractor(xml, tag);
        extractor.extractParametersAndCheckNumberOfParameters(2);
    }

    @Test
    public void extractParametersWithTwoParameters() {
        List<String> xml = CollectionsHelper.buildListFrom(
                "<foo param1=\"1\" param2=\"zwei\">",
                "    <bar>Bar1</bar>",
                "</foo>"
                );
        String tag = "foo";
        XmlParameterExtractor extractor = new XmlParameterExtractor(xml, tag);
        List<NamedXmlParameter> parameters = extractor.extractParameters();

        assertEquals(2, parameters.size());

        NamedXmlParameter param1 = parameters.get(0);
        assertEquals("param1", param1.getName());
        assertEquals("1", param1.getContent());

        NamedXmlParameter param2 = parameters.get(1);
        assertEquals("param2", param2.getName());
        assertEquals("zwei", param2.getContent());
    }

    @Test
    public void extractParametersWithOneParametersAndColSpan() {
        List<String> xml = CollectionsHelper.buildListFrom(
                "<TD colspan=2 class=\"tr6 td34\"><P class=\"p6 ft4\">"
                        + "Nicht mehr Geschäftsführer:</P></TD>"
                );
        String tag = "TD";
        XmlParameterExtractor extractor = new XmlParameterExtractor(xml, tag);
        List<NamedXmlParameter> parameters = extractor.extractParameters();

        assertEquals(2, parameters.size());

        NamedXmlParameter param1 = parameters.get(0);
        assertEquals("colspan", param1.getName());
        assertEquals("2", param1.getContent());

        NamedXmlParameter param2 = parameters.get(1);
        assertEquals("class", param2.getName());
        assertEquals("tr6 td34", param2.getContent());
    }

    @Test
    public void extractSpecialParameter() {
        List<String> xml = CollectionsHelper.buildListFrom(
                "<foo param1=\"1\" param2=\"zwei\">",
                "    <bar>Bar1</bar>",
                "</foo>"
                );
        String tag = "foo";
        XmlParameterExtractor extractor = new XmlParameterExtractor(xml, tag);
        NamedXmlParameter param1 = extractor.extractSpecialParameter("param1");
        assertEquals("param1", param1.getName());
        assertEquals("1", param1.getContent());
    }

    @Test
    public void extractSpecialParameterExpectingOneParameter() {
        List<String> xml = CollectionsHelper.buildListFrom(
                "<foo param1=\"1\">",
                "    <bar>Bar1</bar>",
                "</foo>"
                );
        String tag = "foo";
        XmlParameterExtractor extractor = new XmlParameterExtractor(xml, tag);
        NamedXmlParameter param1 = extractor.extractSpecialParameterAndCheckNumberOfParametersIsOne(
                "param1");
        assertEquals("param1", param1.getName());
        assertEquals("1", param1.getContent());
    }

    @Test (expected = RuntimeException.class)
    public void extractSpecialParameterExpectingOneParameterButThereAreTwo() {
        List<String> xml = CollectionsHelper.buildListFrom(
                "<foo param1=\"1\" param2=\"zwei\">",
                "    <bar>Bar1</bar>",
                "</foo>"
                );
        String tag = "foo";
        XmlParameterExtractor extractor = new XmlParameterExtractor(xml, tag);
        extractor.extractSpecialParameterAndCheckNumberOfParametersIsOne("param1");
    }

    @Test (expected = RuntimeException.class)
    public void extractSpecialParameterButThereIsNoSuchParameter() {
        List<String> xml = CollectionsHelper.buildListFrom(
                "<foo param1=\"1\" param2=\"zwei\">",
                "    <bar>Bar1</bar>",
                "</foo>"
                );
        String tag = "foo";
        XmlParameterExtractor extractor = new XmlParameterExtractor(xml, tag);
        extractor.extractSpecialParameter("param3");
    }

    @Test (expected = RuntimeException.class)
    public void extractSpecialParameterButThereIsNoSuchParameterExpectingOneParameter() {
        List<String> xml = CollectionsHelper.buildListFrom(
                "<foo param1=\"1\">",
                "    <bar>Bar1</bar>",
                "</foo>"
                );
        String tag = "foo";
        XmlParameterExtractor extractor = new XmlParameterExtractor(xml, tag);
        extractor.extractSpecialParameterAndCheckNumberOfParametersIsOne("param3");
    }

}
