package de.duehl.basics.text.xml.generation;

/*
 * Copyright 2019 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import org.junit.Test;

import de.duehl.basics.text.Text;
import de.duehl.basics.text.xml.NamedXmlParameter;

public class XmlBuilderTest {

    @Test
    public void creation() {
        XmlBuilder builder = new XmlBuilder();

        assertNotNull(builder);
    }

    @Test
    public void createAndGetEmptyResult() {
        XmlBuilder builder = new XmlBuilder()
                .appendXmlHeader();

        String actual = builder.toString();
        String n = Text.LINE_BREAK;
        String expected = XmlBuilder.XML_HEADER + n + n;
        assertEquals(expected, actual);
    }

    @Test
    public void appendSomethingClearAndGetEmptyResult() {
        XmlBuilder builder = new XmlBuilder()
                .appendXmlHeader()
                .appendOpeningTag("zeug")
                .clear();

        String actual = builder.toString();
        String expected = "";
        assertEquals(expected, actual);
    }

    @Test
    public void appendSomethingAndGetResult() {
        XmlBuilder builder = new XmlBuilder()
            .appendXmlHeader()
            .appendOpeningTag("zeug")
            .appendInTag("foo", "Wichtiger foo-Inhalt")
            .appendInTag("bar", "Wichtiger bar-Inhalt")

            .appendOpeningTag("baz")
            .appendInTag("baz-foo", "baz-foo-blubb",
                    new NamedXmlParameter("foo", "foo-Inhalt"))
            .appendInTag("baz-bar", "baz-bar-blubb",
                new NamedXmlParameter("foo", "foo-Inhalt"),
                new NamedXmlParameter("bar", "bar-Inhalt"),
                new NamedXmlParameter("baz", "baz-Inhalt"))
            .appendClosingTag("baz")

            .appendOpeningTagWithParameters("open-mit-parameter",
                    new NamedXmlParameter("option", "bla bla"))
            .appendClosingTag("open-mit-parameter")
            .appendClosingTag("zeug");

        String actual = builder.toString();
        String n = Text.LINE_BREAK;

        String expected = XmlBuilder.XML_HEADER + n
                + n
                + "<zeug>" + n
                + "    <foo>Wichtiger foo-Inhalt</foo>" + n
                + "    <bar>Wichtiger bar-Inhalt</bar>" + n
                + "    <baz>" + n
                + "        <baz-foo foo=\"foo-Inhalt\">baz-foo-blubb</baz-foo>" + n
                + "        <baz-bar foo=\"foo-Inhalt\" bar=\"bar-Inhalt\" "
                        + "baz=\"baz-Inhalt\">baz-bar-blubb</baz-bar>" + n
                + "    </baz>" + n
                + "    <open-mit-parameter option=\"bla bla\">" + n
                + "    </open-mit-parameter>" + n
                + "</zeug>" + n
                ;
        assertEquals(expected, actual);
    }

    @Test
    public void appendComment() {
        XmlBuilder builder = new XmlBuilder()
                .appendComment("Kommentar");

        String actual = builder.toString();
        String expected = "<!-- Kommentar -->";
        assertEquals(expected, actual);
    }

    @Test
    public void appendCommentLine() {
        XmlBuilder builder = new XmlBuilder()
                .appendCommentLine("Kommentar");

        String actual = builder.toString();
        String n = Text.LINE_BREAK;
        String expected = "<!-- Kommentar -->" + n;
        assertEquals(expected, actual);
    }

    @Test
    public void appendIndentedCommentLine() {
        XmlBuilder builder = new XmlBuilder()
                .increaseIndentationLevel()
                .appendCommentLine("Kommentar");

        String actual = builder.toString();
        String n = Text.LINE_BREAK;
        String expected = "    <!-- Kommentar -->" + n;
        assertEquals(expected, actual);
    }

    @Test
    public void appendOpeningTableWithClass() {
        XmlBuilder xml = new XmlBuilder()
                .increaseIndentationLevel()
                .increaseIndentationLevel()
                .appendOpeningTagWithParameters("table", "class=\"CSS-Klasse-TABLE\"")
                .appendOpeningTag("tr")
                .appendInTag("th", "Zeug")
                .appendClosingTag("tr")
                .appendOpeningTag("tr")
                .appendInTag("td", "foo")
                .appendClosingTag("tr")
                .appendOpeningTagWithParameters("tr", "class=\"CSS-Klasse-TR\"")
                .appendInTag("td", "bar")
                .appendClosingTag("tr")
                .appendClosingTag("table");

        String actual = xml.toString();
        String n = Text.LINE_BREAK;
        String expected = ""
                + "        <table class=\"CSS-Klasse-TABLE\">" + n
                + "            <tr>" + n
                + "                <th>Zeug</th>" + n
                + "            </tr>" + n
                + "            <tr>" + n
                + "                <td>foo</td>" + n
                + "            </tr>" + n
                + "            <tr class=\"CSS-Klasse-TR\">" + n
                + "                <td>bar</td>" + n
                + "            </tr>" + n
                + "        </table>" + n
                ;
        assertEquals(expected, actual);
    }

}
