package de.duehl.basics.text.xml.own;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.io.FileHelper;
import de.duehl.basics.test.TestHelper;
import de.duehl.basics.text.Text;

public class OwnXmlIOTest {

    private final String path;

    public OwnXmlIOTest() {
        TestHelper testHelper = TestHelper.createForBasics(getClass(), 8);
        path =  testHelper.getTestDataPath() + File.separator;
    }


    /* -------------------------- TESTS -------------------------- */


    @Test
    public void save() {
        String xmlCollectionTag = "obstsorten";
        String xmlDataTag = "obstsorte";
        String filename = path + "own_xml_io_testfile_save.txt";
        List<String> xmlLines = CollectionsHelper.buildListFrom(
                "<obstsorte><name>Apfel</name></obstsorte>",
                "<obstsorte><name>Banane</name></obstsorte>",
                "<obstsorte><name>Orange</name></obstsorte>");

        FileHelper.deleteFileIfExistent(filename);

        OwnXmlIO xmlIo = new OwnXmlIO(xmlCollectionTag, xmlDataTag);
        xmlIo.save(filename, xmlLines);

        String actual = FileHelper.readFileToString(filename);
        String n = Text.LINE_BREAK;
        String expected = "<obstsorten>" + n
                + "    <obstsorte><name>Apfel</name></obstsorte>" + n
                + "    <obstsorte><name>Banane</name></obstsorte>" + n
                + "    <obstsorte><name>Orange</name></obstsorte>" + n
                + "</obstsorten>" + n;
        assertEquals(expected, actual);
    }

    @Test
    public void createSaveLines() {
        String xmlCollectionTag = "obstsorten";
        String xmlDataTag = "obstsorte";
        List<String> xmlLines = CollectionsHelper.buildListFrom(
                "<obstsorte><name>Apfel</name></obstsorte>",
                "<obstsorte><name>Banane</name></obstsorte>",
                "<obstsorte><name>Orange</name></obstsorte>");

        OwnXmlIO xmlIo = new OwnXmlIO(xmlCollectionTag, xmlDataTag);
        List<String> lines = xmlIo.createSaveLines(xmlLines);
        assertEquals(5, lines.size());
        assertEquals("<obstsorten>", lines.get(0));
        assertEquals("    <obstsorte><name>Apfel</name></obstsorte>", lines.get(1));
        assertEquals("    <obstsorte><name>Banane</name></obstsorte>", lines.get(2));
        assertEquals("    <obstsorte><name>Orange</name></obstsorte>", lines.get(3));
        assertEquals("</obstsorten>", lines.get(4));
    }

    @Test
    public void load() {
        String xmlCollectionTag = "obstsorten";
        String xmlDataTag = "obstsorte";
        String filename = path + "own_xml_io_testfile_load.txt";

        List<String> fruits = new ArrayList<>();
        OwnXmlIO xmlIo = new OwnXmlIO(xmlCollectionTag, xmlDataTag);
        xmlIo.load(filename, line -> {
            String name = OwnXmlHelper.readContentOfSingleUniqueXmlElementInXmlLine(
                    "name", line);
            fruits.add(name);
        });
        assertEquals(3, fruits.size());
        assertEquals("Apfel", fruits.get(0));
        assertEquals("Banane", fruits.get(1));
        assertEquals("Orange", fruits.get(2));
    }

    @Test
    public void loadFromVeryLongFile() {
        String xmlCollectionTag = "obstsorten";
        String xmlDataTag = "obstsorte";
        String filename = path + "own_xml_io_testfile_load.txt";

        List<String> fruits = new ArrayList<>();
        OwnXmlIO xmlIo = new OwnXmlIO(xmlCollectionTag, xmlDataTag);
        xmlIo.loadFromVeryLongFile(filename, line -> {
            String name = OwnXmlHelper.readContentOfSingleUniqueXmlElementInXmlLine(
                    "name", line);
            fruits.add(name);
        });
        assertEquals(3, fruits.size());
        assertEquals("Apfel", fruits.get(0));
        assertEquals("Banane", fruits.get(1));
        assertEquals("Orange", fruits.get(2));
    }

    @Test
    public void loadFromXmlLines() {
        String xmlCollectionTag = "obstsorten";
        String xmlDataTag = "obstsorte";
        String filename = path + "own_xml_io_testfile_load.txt";
        List<String> xmlLines = FileHelper.readFileToList(filename);

        List<String> fruits = new ArrayList<>();
        OwnXmlIO xmlIo = new OwnXmlIO(xmlCollectionTag, xmlDataTag);
        xmlIo.loadFromXmlLines(xmlLines, line -> {
            String name = OwnXmlHelper.readContentOfSingleUniqueXmlElementInXmlLine(
                    "name", line);
            fruits.add(name);
        });
        assertEquals(3, fruits.size());
        assertEquals("Apfel", fruits.get(0));
        assertEquals("Banane", fruits.get(1));
        assertEquals("Orange", fruits.get(2));
    }

    @Test
    public void loadFromDirectXmlLines() {
        List<String> xmlLines = CollectionsHelper.buildListFrom(
            "<obstsorten>",
            "    <obstsorte><name>Apfel</name></obstsorte>",
            "    <obstsorte><name>Banane</name></obstsorte>",
            "    <obstsorte><name>Orange</name></obstsorte>",
            "</obstsorten>");

        String xmlCollectionTag = "obstsorten";
        String xmlDataTag = "obstsorte";
        List<String> fruits = new ArrayList<>();
        OwnXmlIO xmlIo = new OwnXmlIO(xmlCollectionTag, xmlDataTag);
        xmlIo.loadFromXmlLines(xmlLines, line -> {
            String name = OwnXmlHelper.readContentOfSingleUniqueXmlElementInXmlLine(
                    "name", line);
            fruits.add(name);
        });
        assertEquals(3, fruits.size());
        assertEquals("Apfel", fruits.get(0));
        assertEquals("Banane", fruits.get(1));
        assertEquals("Orange", fruits.get(2));
    }

    @Test (expected = RuntimeException.class)
    public void loadFromXmlLinesAndFailOnMissindDataTag() {
        List<String> xmlLines = CollectionsHelper.buildListFrom(
            "<obstsorten>",
            "    <abc><name>Apfel</name></abc>",                 // kein Tag "obstsorte" !
            "    <obstsorte><name>Banane</name></obstsorte>",
            "    <obstsorte><name>Orange</name></obstsorte>",
            "</obstsorten>");

        String xmlCollectionTag = "obstsorten";
        String xmlDataTag = "obstsorte";
        OwnXmlIO xmlIo = new OwnXmlIO(xmlCollectionTag, xmlDataTag);
        xmlIo.loadFromXmlLines(xmlLines, line -> {});
    }

}
