package de.duehl.html.download.data;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.logging.Logger;
import de.duehl.html.download.proxy.Proxy;
import de.duehl.html.download.proxy.RotatingProxies;

/**
 * Diese Klasse stellt die Parameter für den Download dar, die der Downloader mit seinen Methoden
 * beim Start des Downloads zusammensetzt.
 *
 * @version 1.01     2017-05-18
 * @author Christian Dühl
 */

public class DownloadParameters {

    /** Die Webseite, die hier heruntergeladen werden soll. */
    private final String website;

    /** Gibt an, wie mit Redirects umgegangen werden soll. */
    private RedirectHandling redirectHandling;

    /** Gibt an, ob ein Proxy verwendet werden soll. */
    private boolean useProxy;

    /** Der zu verwendende Proxy, wenn einer verwendet werden soll. */
    private Proxy proxy;

    /** Gibt an, ob ein rotierender Proxy verwendet werden soll. */
    private boolean useRotatingProxies;

    /** Der zu verwendende rotierende Proxy, wenn einer verwendet werden soll. */
    private RotatingProxies rotatingProxies;

    /**
     * Maximale zulässige Anzahl der Versuche, eine Seite herunter zu laden. Falls ein rotierender
     * Proxy verwendet wird, ist dies die Anzahl der Versuche, die Seite mit jedem Proxy herunter
     * zu laden.
     */
    private int maximumNumberOfTries;

    /** Schlafenszeit zwischen den Versuchen. */
    private long secondsToSleep;

    /** Der ggf. verwendete Logger. */
    private Logger logger;

    /**
     * Gibt an, ob ein TimedThreadRunner verwendet werden soll, um den Download nach einer
     * bestimmten Zeit abzuwürgen.
     */
    private boolean useTimedThreadRunner;

    /** Wenn secondsBeforeKill oder mehr Sekunden vergangen sind, wird der Download beendet. */
    private int secondsBeforeKill;

    /**
     * Alle secondsBetweenWatching Sekunden wird der Thread, in dem die Aufgabe läuft, auf
     * Zeitüberschreitung überprüft.
     */
    private long millisecondsBetweenWatching;

    /**
     * Konstruktor.
     *
     * @param website
     *            Die Webseite, die hier heruntergeladen werden soll.
     */
    public DownloadParameters(String website) {
        this.website = website;

        redirectHandling = RedirectHandling.STANDARD;

        useProxy = false;
        maximumNumberOfTries = 1;
        useTimedThreadRunner = false;
        secondsBeforeKill = -1;
        millisecondsBetweenWatching = -1;
    }

    /** Getter für die Webseite, die hier heruntergeladen werden soll. */
    public String getWebsite() {
        return website;
    }

    /** Gibt an, wie mit Redirects umgegangen werden soll. */
    public RedirectHandling getRedirectHandling() {
        return redirectHandling;
    }

    /** Gibt an, ob ein Proxy verwendet werden soll. */
    public boolean useProxy() {
        return useProxy;
    }

    /** Getter für den zu verwendenden Proxy, wenn einer verwendet werden soll. */
    public Proxy getProxy() {
        return proxy;
    }

    /** Gibt an, ob ein rotierender Proxy verwendet werden soll. */
    public boolean useRotatingProxies() {
        return useRotatingProxies;
    }

    /** Getter für den zu verwendenden rotierende Proxy, wenn einer verwendet werden soll. */
    public RotatingProxies getRotatingProxies() {
        return rotatingProxies;
    }

    /**
     * Getter für die maximale zulässige Anzahl der Versuche, eine Seite herunter zu laden. Falls
     * ein rotierender Proxy verwendet wird, ist dies die Anzahl der Versuche, die Seite mit jedem
     * Proxy herunter zu laden.
     */
    public int getMaximumNumberOfTries() {
        return maximumNumberOfTries;
    }

    /** Getter für die Schlafenszeit zwischen den Versuchen. */
    public long getSecondsToSleep() {
        return secondsToSleep;
    }

    /** Gibt an, ob wir einen Logger zur Verfügung haben. */
    public boolean weHaveALogger() {
        return logger != null;
    }

    /** Getter für den ggf. verwendete Logger. */
    public Logger getLogger() {
        return logger;
    }

    /**
     * Gibt an, ob ein TimedThreadRunner verwendet werden soll, um den Download nach einer
     * bestimmten Zeit abzuwürgen.
     */
    public boolean useTimedThreadRunner() {
        return useTimedThreadRunner;
    }

    /**
     * Getter: Wenn secondsBeforeKill oder mehr Sekunden vergangen sind, wird der Download beendet.
     */
    public int getSecondsBeforeKill() {
        return secondsBeforeKill;
    }

    /**
     * Getter: Alle secondsBetweenWatching Sekunden wird der Thread, in dem die Aufgabe läuft, auf
     * Zeitüberschreitung überprüft.
     */
    public long getMillisecondsBetweenWatching() {
        return millisecondsBetweenWatching;
    }

    /** Legt fest, wie mit Redirects umgegangen werden soll. */
    public void setRedirectHandling(RedirectHandling redirectHandling) {
        this.redirectHandling = redirectHandling;
    }

    /** Legt fest, ob ein Proxy verwendet werden soll. */
    public void setUseProxy(boolean useProxy) {
        this.useProxy = useProxy;
    }

    /** Setter für den zu verwendenden Proxy, wenn einer verwendet werden soll. */
    public void setProxy(Proxy proxy) {
        this.proxy = proxy;
    }

    /** Legt fest, ob ein rotierender Proxy verwendet werden soll. */
    public void setUseRotatingProxies(boolean useRotatingProxies) {
        this.useRotatingProxies = useRotatingProxies;
    }

    /** Setter für den zu verwendenden rotierende Proxy, wenn einer verwendet werden soll. */
    public void setRotatingProxies(RotatingProxies rotatingProxies) {
        this.rotatingProxies = rotatingProxies;
    }

    /**
     * Setter für die maximale zulässige Anzahl der Versuche, eine Seite herunter zu laden. Falls
     * ein rotierender Proxy verwendet wird, ist dies die Anzahl der Versuche, die Seite mit jedem
     * Proxy herunter zu laden.
     */
    public void setMaximumNumberOfTries(int maximumNumberOfTries) {
        this.maximumNumberOfTries = maximumNumberOfTries;
    }

    /** Setter für die Schlafenszeit zwischen den Versuchen. */
    public void setSecondsToSleep(long secondsToSleep) {
        this.secondsToSleep = secondsToSleep;
    }

    /** Setter für den ggf. verwendete Logger. */
    public void setLogger(Logger logger) {
        this.logger = logger;
    }

    /**
     * Legt fest, ob ein TimedThreadRunner verwendet werden soll, um den Download nach einer
     * bestimmten Zeit abzuwürgen.
     */
    public void setUseTimedThreadRunner(boolean useTimedThreadRunner) {
        this.useTimedThreadRunner = useTimedThreadRunner;
    }

    /**
     * Setter: Wenn secondsBeforeKill oder mehr Sekunden vergangen sind, wird der Download beendet.
     */
    public void setSecondsBeforeKill(int secondsBeforeKill) {
        this.secondsBeforeKill = secondsBeforeKill;
    }

    /**
     * Setter: Alle secondsBetweenWatching Sekunden wird der Thread, in dem die Aufgabe läuft, auf
     * Zeitüberschreitung überprüft.
     */
    public void setMillisecondsBetweenWatching(long millisecondsBetweenWatching) {
        this.millisecondsBetweenWatching = millisecondsBetweenWatching;
    }

    @Override
    public String toString() {
        return "DownloadParameters [website=" + website + ", redirectHandling=" + redirectHandling
                + ", useProxy=" + useProxy + ", proxy=" + proxy + ", useRotatingProxies="
                + useRotatingProxies + ", rotatingProxies=" + rotatingProxies
                + ", maximumNumberOfTries=" + maximumNumberOfTries + ", secondsToSleep="
                + secondsToSleep + ", logger=" + logger + ", useTimedThreadRunner="
                + useTimedThreadRunner + ", secondsBeforeKill=" + secondsBeforeKill
                + ", millisecondsBetweenWatching=" + millisecondsBetweenWatching + "]";
    }

}
