package de.duehl.swing.ui.buttons.creator;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Color;
import java.awt.Component;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.BorderFactory;
import javax.swing.JLabel;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.colors.ColorTool;

/**
 * Diese Klasse erstellt Labels mit farbigem Text, welche als Buttons verwendet werden.
 *
 * @version 1.01     2021-05-07
 * @author Christian Dühl
 */

public class ColoredLabelButtonCreator implements ButtonCreator {

    /** Textfarbe des Buttons. */
    private final Color labelColor;

    /**
     * Gibt an ob eine Textfarbe des Buttons, wenn man mit der Maus über den Button fährt,
     * spezifiziert wurde.
     */
    private boolean mouseEnteredLabelColorSpecified;

    /** Textfarbe des Buttons, wenn man mit der Maus über den Button fährt. */
    private Color mouseEnteredLabelColor;

    /**
     * Gibt an ob eine Textfarbe des Buttons, wenn man mit der Maus den Button drückt aber noch
     * nicht losgelassen hat, spezifiziert wurde.
     */
    private boolean mousePressedLabelColorSpecified;

    /**
     * Textfarbe des Buttons, wenn man mit der Maus den Button drückt aber noch nicht losgelassen
     * hat.
     */
    private Color mousePressedLabelColor;

    /**
     * Konstruktor.
     *
     * @param labelColor
     *            Textfarbe des Buttons.
     */
    public ColoredLabelButtonCreator(Color labelColor) {
        this.labelColor = labelColor;
        mouseEnteredLabelColorSpecified = false;
        mousePressedLabelColorSpecified = false;
    }

    /**
     * Legt die Textfarbe des Buttons fest, wenn man mit der Maus den Button drückt aber noch nicht
     * losgelassen hat.
     *
     * Standardmäßig wird die Weiß verwendet.
     *
     * @param mousePressedLabelColor
     *            Textfarbe die die Beschriftung des Buttons haben soll, wenn man mit der Maus den
     *            Button drückt aber noch nicht losgelassen hat.
     * @return Gibt das Objekt ColoredLabelButtonCreator zurück, damit man diesen Setter direkt an
     *         new anhängen und ihn dabei einer Variablen zuweisen kann.
     */
    public ColoredLabelButtonCreator setMousePressedLabelColor(Color mousePressedLabelColor) {
        mousePressedLabelColorSpecified = true;
        this.mousePressedLabelColor = mousePressedLabelColor;
        return this;
    }

    /**
     * Legt die Textfarbe des Buttons fest, wenn man mit der Maus über den Button fährt.
     *
     * Standardmäßig wird die entgegengesetzte Farbe verwendet.
     *
     * @param mouseEnteredLabelColor
     *            Textfarbe die die Beschriftung des Buttons haben soll, wenn man mit der Maus über
     *            den Button fährt.
     * @return Gibt das Objekt ColoredLabelButtonCreator zurück, damit man diesen Setter direkt an
     *         new anhängen und ihn dabei einer Variablen zuweisen kann.
     */
    public ColoredLabelButtonCreator setMouseEnteredLabelColor(Color mouseEnteredLabelColor) {
        mouseEnteredLabelColorSpecified = true;
        this.mouseEnteredLabelColor = mouseEnteredLabelColor;
        return this;
    }

    /**
     * Erstellt einen Button mit der passenden Aufschrift und hinterlegt die Aktion für den Klick
     * auf den Button.
     *
     * @param text
     *            Beschriftung des Buttons.
     * @param buttonClickAction
     *            Aktion die bei einem Klick auf den Button ausgeführt wird.
     */
    @Override
    public Component createButton(String text, Runnable buttonClickAction) {
        JLabel label = new JLabel();

        label.setText(text);
        GuiTools.boldFont(label);
        GuiTools.biggerFont(label, 5);
        label.setHorizontalAlignment(JLabel.CENTER);
        label.setVerticalAlignment(JLabel.CENTER);
        label.setForeground(labelColor);
        makeLabelClickable(label, buttonClickAction);
        addEmptyBorderArroundLabel(label);

        return label;
    }

    private void makeLabelClickable(JLabel label, Runnable buttonClickAction) {
        label.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent e) {
                buttonClickAction.run();
            }
            @Override
            public void mouseEntered(MouseEvent e) {
                label.setForeground(createMouseEnteredLabelColor());
            }
            @Override
            public void mouseExited(MouseEvent e) {
                label.setForeground(labelColor);
            }
            @Override
            public void mousePressed(MouseEvent e) {
                label.setForeground(createMousePressedLabelColor());
            }
            @Override
            public void mouseReleased(MouseEvent e) {
                label.setForeground(createMouseEnteredLabelColor());
            }
        });
    }

    private Color createMouseEnteredLabelColor() {
        if (mouseEnteredLabelColorSpecified) {
            return mouseEnteredLabelColor;
        }
        else {
            Color enteredColor = ColorTool.anticolor(labelColor);
            enteredColor = ColorTool.changeColor(enteredColor, -50);
            // Color.MAGENTA
            // new Color(200, 0, 255)
            return enteredColor;
        }
    }

    private Color createMousePressedLabelColor() {
        if (mousePressedLabelColorSpecified) {
            return mousePressedLabelColor;
        }
        else {
            return Color.WHITE;
        }
    }

    private void addEmptyBorderArroundLabel(JLabel label) {
        label.setBorder(BorderFactory.createEmptyBorder(2, 5, 2, 5));
    }

}
