package de.duehl.swing.ui.buttons.painted.state;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.swing.ui.geometry.PixelPoint;

/**
 * Diese Klasse stellt einen kleinen Schalter dar, der einen gefüllten Kreis darstellt.
 *
 * Er kann zwei Zustände darstellen, ausgewählt und nicht ausgewählt. Ist der Zustand aktiviert,
 * wird das Kreis ausgefüllt, anderenfalls ist nur dessen Rand zu sehen.
 *
 * @version 1.01     2025-11-21
 * @author Christian Dühl
 */

final public class CircleButton extends PaintedStateButton {

    private static final long serialVersionUID = 1L;

    /** Zeichnet das Symbol in horizontaler und rechteckiger Anordnung. */
    @Override
    protected void paintHorizontalSymbolRectangle() {
        paintSymbol();
    }

    /** Zeichnet das Symbol in vertikaler und rechteckiger Anordnung. */
    @Override
    protected void paintVerticalSymbolRectangle() {
        paintSymbol();
    }

    /** Zeichnet das Symbol in horizontaler und quadratischer Anordnung. */
    @Override
    protected void paintHorizontalSymbolSquare() {
        paintSymbol();
    }

    /** Zeichnet das Symbol in vertikaler und quadratischer Anordnung. */
    @Override
    protected void paintVerticalSymbolSquare() {
        paintSymbol();
    }

    /**
     * Zeichnet den Kreis. Da es dabei egal ist, ob horizontal oder vertikal, wird
     * es in beiden Fällen verwendet.
     */
    private void paintSymbol() {
        int width = getWidth();
        int height = getHeight();

        if (isForceSquare()) {
            width = Math.min(width, height);
            height = width;
        }

        int delta = 6;

        int firstLeftPixel = (getWidth() - width)/2;
        int left = firstLeftPixel + delta;

        int lastRightPixel = firstLeftPixel + width - 1;
        int right = lastRightPixel - delta;

        int firstUpperPixel = (getHeight() - height)/2;
        int upper = firstUpperPixel + delta;

        int lastLowerPixel = firstUpperPixel + height - 1;
        int lower = lastLowerPixel - delta;

        int innerWidth = right - left;
        int innerHeight = lower - upper;
        int centerX = left + innerWidth / 2;
        int centerY = upper + innerHeight / 2;
        PixelPoint center = new PixelPoint(centerX, centerY);


        int halfSmallerWidthOrHeight = Math.min(width, height) / 2;
        int radius = halfSmallerWidthOrHeight - delta;

        if (isElected()) {
            paintCircle(center, radius);
        }
        else {
            paintHollowCircle(center, radius);
        }
    }

}
