package de.duehl.swing.ui.components.selections;

/*
 * Copyright 2019 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ButtonGroup;
import javax.swing.JPanel;
import javax.swing.JRadioButton;

import de.duehl.basics.io.Charset;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.layout.VerticalLayout;

/**
 * Diese Klasse stellt eine Auswahl der Charsets zur Verfügung.
 *
 * @version 1.01     2019-11-05
 * @author Christian Dühl
 */

public class CharsetSelection {

    private static final int VERTICAL_SPACE = 2;

    private ButtonGroup group;
    private List<JRadioButton> charsetButtons;

    /** Der hier erzeugte Panel mit den Charset-Buttons. */
    private JPanel panel;

    public CharsetSelection(String title) {
        group = new ButtonGroup();
        charsetButtons = new ArrayList<>();

        for (Charset charset : Charset.values()) {
            charsetButtons.add(new JRadioButton(charset.getCharsetAsString()));
        }
        select(Charset.UTF_8);

        panel = new JPanel();
        panel.setLayout(new VerticalLayout(VERTICAL_SPACE, VerticalLayout.BOTH));
        GuiTools.createTitle(title, panel);

        addToGroup();
        addToPanel();
    }

    private void addToGroup() {
        for (JRadioButton radioButton : charsetButtons) {
            group.add(radioButton);
        }
    }

    private void addToPanel() {
        for (JRadioButton radioButton : charsetButtons) {
            panel.add(radioButton);
        }
    }

    /** Getter für den hier erzeugten Panel mit den Charset-Buttons. */
    public JPanel getPanel() {
        return panel;
    }

    /** Legt die Auswahl auf das übergebene Charset fest. */
    public void select(Charset charset) {
        String text = charset.getCharsetAsString();
        for (JRadioButton radioButton : charsetButtons) {
            String buttonText = radioButton.getText();
            radioButton.setSelected(buttonText.equals(text));
        }
    }

    /** Ermittelt das ausgewählte Charset. */
    public Charset getSelected() {
        for (JRadioButton radioButton : charsetButtons) {
            if (radioButton.isSelected()) {
                String buttonText = radioButton.getText();
                return Charset.detectFromCharsetAsString(buttonText);
            }
        }

        throw new RuntimeException("Kein Button ausgewählt.");
    }

    /** Setzt den Tooltip. */
    public void setToolTipText(String tooltip) {
        panel.setToolTipText(tooltip);
        for (JRadioButton button : charsetButtons) {
            button.setToolTipText(tooltip);
        }
    }

    /** Fügt ActionListener zu den Buttons hinzu. */
    public void addActionListener(ActionListener listener) {
        for (JRadioButton button : charsetButtons) {
            button.addActionListener(listener);
        }
    }

}
