package de.duehl.swing.ui.components.selections.base;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import de.duehl.basics.text.Text;
import de.duehl.swing.ui.colors.Colorizer;
import de.duehl.swing.ui.components.elements.TextFieldWithChangeButtonAndOption;

/**
 * Diese Klasse stellt die abstrakte Basis für eine optionale Auswahl einer Datei oder eines
 * Verzeichnisses etwa in einem Optionendialog dar.
 *
 * @version 1.01     2017-03-14
 * @author Christian Dühl
 */

public abstract class OptionalFileOrPathSelection {

    /** Optionaler Button, Textfeld und Änderungs-Schalter. */
    private final TextFieldWithChangeButtonAndOption elements;

    /**
     * Konstruktor.
     *
     * @param title
     *            Überschrift für die Wahl dieser Datei oder dieses Verzeichnisses.
     */
    public OptionalFileOrPathSelection(String title) {
        elements = new TextFieldWithChangeButtonAndOption(title);
        elements.addButtonActionListener(createActionListener());
    }

    private ActionListener createActionListener() {
        return new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent arg0) {
                changeDirectory();
            }
        };
    }

    /** Wird aufgerufen, wenn der Button zum Ändern des Pfades gedrückt wird. */
    private void changeDirectory() {
        String selection = openPath(elements.getText());
        if (!selection.isEmpty()) {
            elements.setText(Text.nicePath(selection));
        }
    }

    /**
     * Der Benutzer wählt die Datei oder das Verzeichnis aus.
     *
     * @param startDir
     *            Startverzeichnis.
     * @return Ausgewählte Benutzereingabe oder null bei Abbruch durch den Benutzer.
     */
    abstract protected String openPath(String startDir);

    /** Fragt ab, ob die Option ausgewählt ist. */
    public boolean isSelected() {
        return elements.isSelected();
    }

    /** Setzt die Option auf ausgewählt oder nicht ausgewählt. */
    public void setSelected(boolean selected) {
        elements.setSelected(selected);
    }

    /** Getter für das Panel mit Überschrift, Pfad und Änderungsbutton. */
    public Component getPanel() {
        return elements.getPanel();
    }

    /** Färbt alle Komponenten mit dem übergebenen Colorizer ein. */
    public void colorize(Colorizer colorizer) {
        elements.colorize(colorizer);
    }

    /** Getter für den (getrimmten) Text des Textfeldes. */
    public String getText() {
        return elements.getText();
    }

    /** Setzt den Text des Textfeldes. */
    public void setText(String text) {
        elements.setText(text);
    }

    /** Setter für die gewünschte Größe des Textfeldes. */
    public void setPreferredSize(Dimension preferredSize) {
        elements.setPreferredSize(preferredSize);
    }

}
