package de.duehl.swing.ui.dialogs.logging;

/*
 * Copyright 2022 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.ArrayList;
import java.util.List;

import javax.swing.table.AbstractTableModel;

import de.duehl.basics.logging.LogEntry;

/**
 * Diese Klasse ist ein eigenes Tabellenmodell. Es hält die Daten der Tabelle in Form einer Liste
 * von LogFileTableEntry Objekten vor und weiß diese anzuzeigen. Darüber hinaus kann man die
 * Tabelle leeren und Daten anhängen.
 *
 * @version 1.01     2022-07-20
 * @author Christian Dühl
 */

public class LogFileTableModel extends AbstractTableModel {

    private static final long serialVersionUID = 1L;

    /** Liste mit den Elementen der Tabelle: */
    private final List<LogEntry> entryList;

    /** Konstruktor. */
    public LogFileTableModel() {
        super();
        entryList = new ArrayList<>();
    }

    /** Ermittelt die Spaltenzahl. */
    @Override
    public int getColumnCount() {
        return 6;
    }

    /** Ermittelt die Zeilenzahl. */
    @Override
    public int getRowCount() {
        return entryList.size();
    }

    /**
     * Ermittelt den Tabelleninhalt der angegebenen Zelle.
     *
     * @param row
     *            Zeilennummer (die erste hat die 0)
     * @param column
     *            Spaltennummer (die erste hat die 0)
     */
    @Override
    public Object getValueAt(int row, int column) {
        String value = "";
        LogEntry entry = entryList.get(row);
        switch (column) {
            case 0:
                value = entry.getDate();
                break;
            case 1:
                value = entry.getTime();
                break;
            case 2:
                value = entry.getClassName();
                break;
            case 3:
                value = entry.getMethod();
                break;
            case 4:
                value = entry.getLineNumber();
                break;
            case 5:
                value = entry.getMessage();
                break;
        }
        return value;
    }

    /** Löscht den Inhalt der Tabelle. */
    public void clear() {
        entryList.clear();
    }

    /**
     * Fügt ein Element der Tabelle hinzu.
     *
     * @param entry
     *            Neuer Tabelleneintrag
     */
    public void addEntry(LogEntry entry) {
        entryList.add(entry);
    }

    /** Getter für das Element an dem gegebenen Zeilenindex (0-basiert). */
    public LogEntry getRow(int row) {
        return entryList.get(row);
    }

    /** Getter für die Liste alle Zeilen. */
    public List<LogEntry> getRows() {
        List<LogEntry> copied = new ArrayList<>();
        copied.addAll(entryList);
        return copied;
    }

}
