package de.duehl.swing.ui.elements.slider.data;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.system.SystemTools;

/**
 * Diese Klasse aktualisiert, solange sie (in einem eigenen Thread) läuft, etwas regelmäßig.
 *
 * @version 1.01     2025-07-24
 * @author Christian Dühl
 */

public class UpdateRunnable implements Runnable {

    /** Default Zeitraum in Millisekunden zwischen Aktualisierungen. */
    private final static int SLEEP_TIME = 50;


    /** Gibt an, ob die Aktualisierung des Labels läuft. */
    private volatile boolean updateRunning;

    /** Wird ausgeführt, wenn aktualisiert werden soll. */
    private final Runnable updateRunnable;

    /** Zeitraum in Millisekunden zwischen Aktualisierungen. */
    private volatile long sleepTimeMillis;

    /**
     * Konstruktor.
     *
     * @param actualisationRunnable
     *            Wird ausgeführt, wenn aktualisiert werden soll.
     */
    public UpdateRunnable(Runnable actualisationRunnable) {
        this(actualisationRunnable, SLEEP_TIME);
    }

    /**
     * Konstruktor.
     *
     * @param updateRunnable
     *            Wird ausgeführt, wenn aktualisiert werden soll.
     */
    public UpdateRunnable(Runnable updateRunnable, long sleepTimeMillis) {
        this.updateRunnable = updateRunnable;
        this.sleepTimeMillis = sleepTimeMillis;
    }

    /**
     * Setzt die Millisekunden zwischen den Aktualisierungen des StopWatchLabels. Default ist 50ms.
     */
    public void setSleepTimeMillis(int sleepTimeMillis) {
        this.sleepTimeMillis = sleepTimeMillis;
    }

    /** Führt den Aktualisierungs-Task aus. */
    @Override
    public void run() {
        updateRunning = true;
        loop();
    }

    private void loop() {
        while (updateRunning) {
            updateRunnable.run();
            SystemTools.sleep(sleepTimeMillis);
        }
    }

    /** Beendet die Aktualisierungen. */
    public void stop() {
        updateRunning = false;
    }

}
