package de.duehl.swing.ui.io.lock;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.io.lock.FileLock;

/**
 * Diese Klasse kümmert sich um eine Lockdatei, welche verhindert, dass mehrere Programmaufrufe zur
 * gleichen Zeit stattfinden.
 *
 * Die Klasse de.duehl.basics.io.lock.FileLock wird dabei um eine bequeme Methode zum
 * wiederholten Versuchen das Lock zu erhalten ergänzt.
 *
 * @version 1.01     2021-02-09
 * @author Christian Dühl
 */

public class InteractiveProgramFileLock extends InteractiveProgramLock<FileLock> {

    /**
     * Konstruktor der das Lesen und Schreiben lockt (Rechte "rw").
     *
     * @param filename
     *            Name der Sperrdatei, die einen mehrfachen Start verhindert.
     */
    public InteractiveProgramFileLock(String filename) {
        super(new FileLock(filename));
    }

    /**
     * Konstruktor.
     *
     * @param filename
     *            Name der Sperrdatei, die einen mehrfachen Start verhindert.
     * @param rights
     *            Rechte für das Lock:                                                     <ul><li>
     *            "r"   Open for reading only.                                            </li><li>
     *            "rw"  Open for reading and writing.                                     </li><li>
     *            "rws" Open for reading and writing, as with "rw", and
     *                  also require that every update to the file's content
     *                  or metadata be written synchronously to the underlying
     *                  storage device.                                                   </li><li>
     *            "rwd" Open for reading and writing, as with "rw", and also
     *                  require that every update to the file's content be written
     *                  synchronously to the underlying storage device.                  </li></ul>
     */
    public InteractiveProgramFileLock(String filename, String rights) {
        super(new FileLock(filename, rights));
    }

    @Override
    protected String createNoLockGainedDialogTitle() {
        return "Die zu lockende Datei ist bereits gelockt!";
    }

    @Override
    protected String createNoLockGainedDialogMessage() {
        return "Die zu lockende Datei\n\n"
                + "\t" + getFilename() + "\n\n"
                + " ist bereits gelockt!\n\n"
                + "Bitte entscheiden Sie, wie weiter vogegangen werden soll. "
                + "Die zur Verfügung stehenden Möglichkeiten lauten:\n"
                + "\t" + "1) Abbruch des Programms" + "\n"
                + "\t" + "2) Lock entfernen und den Schritt wiederholen" + "\n"
                + "\t" + "3) Schritt wiederhoilen, ohne den Lock zu entfernen" + "\n"
                + "Achtung beim Abbruch geht eventuell ungespeicherte Arbeit verloren.";
    }

}
