package de.duehl.swing.ui.io.lock.ui;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Image;
import java.awt.Point;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import de.duehl.basics.text.html.HtmlTool;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;

/**
 * Diese Klasse öffnet einen Dialog zur Anzeige, dass ein Lock nicht erhalten werden konnte.
 * Es gibt drei Auswahlmöglichkeiten:
 *     - Erneuter Versuch das Lock zu erhalten ohne dieses vorher zu entfernen
 *     - Entfernen des Locks und Wiederholung des Schrittes.
 *     - Abbruch des Programms.
 * Abbrechen des Dialogs gilt als Abbruch.
 *
 * @version 1.01     2021-02-09
 * @author Christian Dühl
 */

public class NoLockGainedDialog extends ModalDialogBase {

    /** Nachricht des Dialogs. */
    private final String message;

    /** Antwort des Benutzers. */
    private NoLockGainedDialogAnswer answer;

    private final JButton retryWithoutRemovingLockButton;
    private final JButton removeLockAndRetryButton;
    private final JButton cancelButton;

    /**
     * Konstruktor.
     *
     * @param title
     *            Titel des Dialogs.
     * @param message
     *            Nachricht des Dialogs.
     */
    public NoLockGainedDialog(String title, String message) {
        this(new Point(200, 100), null, title, message);
    }

    /**
     * Konstruktor.
     *
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog
     *            erzeugt wird.
     * @param programImage
     *            Image für den Dialog.
     * @param title
     *            Titel des Dialogs.
     * @param message
     *            Nachricht des Dialogs.
     */
    public NoLockGainedDialog(Point parentLocation, Image programImage, String title,
            String message) {
        super(parentLocation, programImage, title);
        addClosingWindowListener(() -> cancel());

        this.message = message;

        answer = NoLockGainedDialogAnswer.CANCEL;

        retryWithoutRemovingLockButton = createRetryWithoutRemovingLockButton();
        removeLockAndRetryButton = createRemoveLockAndRetryButton();
        cancelButton = createCancelButton();

        fillDialog();

        SwingUtilities.invokeLater(() -> retryWithoutRemovingLockButton.requestFocusInWindow());
    }

    private JButton createRetryWithoutRemovingLockButton() {
        JButton button = new JButton("Erneut versuchen");
        GuiTools.boldFont(button);
        //button.requestFocusInWindow();
        button.addActionListener(e -> retryWithoutRemovingLock());
        GuiTools.biggerFont(button, 2);

        return button;
    }

    private JButton createRemoveLockAndRetryButton() {
        JButton button = new JButton("Lock entfernen und wiederholen");
        button.addActionListener(e -> removeLockAndRetry());
        GuiTools.biggerFont(button, 2);

        return button;
    }

    private JButton createCancelButton() {
        JButton button = new JButton("Abbruch");
        button.addActionListener(e -> cancel());
        GuiTools.biggerFont(button, 2);

        return button;
    }

    @Override
    protected void populateDialog() {
        add(createMessagePart(), BorderLayout.CENTER);
        add(createButtonPart(), BorderLayout.SOUTH);

        addEscapeBehaviour();
    }

    protected JLabel createMessagePart() {
        JLabel label = new JLabel(HtmlTool.htmlify(message));
        GuiTools.biggerFont(label, 5);
        label.setBorder(BorderFactory.createEmptyBorder(3, 5, 3, 5));
        GuiTools.createTitle(label);
        return label;
    }

    private Component createButtonPart() {
        JPanel panel = new JPanel();
        setColors(panel);
        panel.setLayout(new BorderLayout());

        panel.add(createButtons(), BorderLayout.EAST);
        panel.add(cancelButton, BorderLayout.WEST);

        return panel;
    }

    private Component createButtons() {
        JPanel panel = new JPanel();
        setColors(panel);
        panel.setLayout(new BorderLayout());

        panel.add(retryWithoutRemovingLockButton, BorderLayout.EAST);
        panel.add(removeLockAndRetryButton, BorderLayout.WEST);

        return panel;
    }

    private void retryWithoutRemovingLock() {
        answer = NoLockGainedDialogAnswer.RETRY_WITHOUT_REMOVING_LOCK;
        closeDialog();
    }

    private void removeLockAndRetry() {
        answer = NoLockGainedDialogAnswer.REMOVE_LOCK_AND_RETRY;
        closeDialog();
    }

    private void cancel() {
        answer = NoLockGainedDialogAnswer.CANCEL;
        closeDialog();
    }

    /** Getter für die Antwort des Benutzers. */
    public NoLockGainedDialogAnswer getAnswer() {
        return answer;
    }

}
