package de.duehl.swing.ui.tabs.bars.generic;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Component;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Diese Klasse stellt ein Reiterelement dar, welches eine Reihe von festen, benannten Komponenten
 * aufnimmt.
 *
 * Dabei wird in jeden Reiter oben ein eigenes Tabulator-Objekt eingefügt, das anzeigen kann, ob
 * der Inhalt des Reiters geändert wurde oder unverändert ist.
 *
 * Im Gegensatz zur TabBar speichert dieses Reiterelement die Klassen, welche die Inhalte der
 * Reiter erzeugen.
 *
 * @param <Creator>
 *            Die Klasse Creator steht für den Typ der Klassen, die die Inhalte in den Reitern
 *            liefern.
 *
 * @version 1.01     2021-11-30
 * @author Christian Dühl
 */

public class StoringTabBar<Creator extends ComponentCreator> extends TabBar<Component> {

    /** Liste der Titel. */
    private final List<String> titles;

    /** Verzeichnis der Inhaltserzeugern nach den Titeln. */
    private final Map<String, Creator> creatorsByTitles;

    /**
     * Konstruktor.
     *
     * @param showCloseButton
     *            Gibt an, ob der Button zum Löschen angezeigt werden soll.
     */
    public StoringTabBar(boolean showCloseButton) {
        super(showCloseButton);
        titles = new ArrayList<>();
        creatorsByTitles = new HashMap<>();
    }

    /**
     * Fügt einen Tabulator hinzu.
     *
     * @param title
     *            Titel des Reiters.
     * @param creator
     *            Inhaltserzeuger.
     */
    public void addTab(String title, Creator creator) {
        super.addTab(title, creator.createComponent());
        titles.add(title);
        creatorsByTitles.put(title, creator);
    }

    /** Entfernt den Reiter mit dem angegebenen Namen. */
    @Override
    public void removeTab(String title) {
        super.removeTab(title);
        creatorsByTitles.remove(title);
        titles.remove(title);
    }

    /** Entfernt alle Reiter. */
    @Override
    public void removeAllTabs() {
        super.removeAllTabs();
        creatorsByTitles.clear();
        titles.clear();
    }

    /** Erzeugt eine Liste der Inhaltserzeuger. */
    public List<Creator> getCreators() {
        List<Creator> list = new ArrayList<>();
        for (String title : titles) {
            list.add(creatorsByTitles.get(title));
        }
        return list;
    }

}
