package de.duehl.swing.persistance;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;

import static org.junit.Assert.*;

public class SwingSessionManagerHelperTest {

    @Test
    public void createPointStorageStringZeroZero() {
        Point point = new Point();
        String actual = SwingSessionManagerHelper.createPointStorageString(point);
        String expected = "0#0";
        assertEquals(expected, actual);
    }

    @Test
    public void createPointStorageString() {
        Point point = new Point(317, 115);
        String actual = SwingSessionManagerHelper.createPointStorageString(point);
        String expected = "317#115";
        assertEquals(expected, actual);
    }

    @Test
    public void createPointFromStorageStringZeroZero() {
        String storageString = "0#0";
        Point actual = SwingSessionManagerHelper.createPointFromStorageString(storageString);
        Point expected = new Point();
        assertEquals(expected, actual);
    }

    @Test
    public void createPointFromStorageString() {
        String storageString = "317#115";
        Point actual = SwingSessionManagerHelper.createPointFromStorageString(storageString);
        Point expected = new Point(317, 115);
        assertEquals(expected, actual);
    }

    @Test
    public void createPointFromGarbageStorageString() {
        String storageString = "317#115.7";
        Point actual = SwingSessionManagerHelper.createPointFromStorageString(storageString);
        Point expected = SwingSessionManagerHelper.NOT_INITIALISED_POINT;
        assertEquals(expected, actual);
    }

    @Test
    public void createDimensionStorageStringZeroZero() {
        Dimension point = new Dimension();
        String actual = SwingSessionManagerHelper.createDimensionStorageString(point);
        String expected = "0#0";
        assertEquals(expected, actual);
    }

    @Test
    public void createDimensionStorageString() {
        Dimension point = new Dimension(317, 115);
        String actual = SwingSessionManagerHelper.createDimensionStorageString(point);
        String expected = "317#115";
        assertEquals(expected, actual);
    }

    @Test
    public void createDimensionFromStorageStringZeroZero() {
        String storageString = "0#0";
        Dimension actual = SwingSessionManagerHelper.createDimensionFromStorageString(storageString);
        Dimension expected = new Dimension();
        assertEquals(expected, actual);
    }

    @Test
    public void createDimensionFromStorageString() {
        String storageString = "317#115";
        Dimension actual = SwingSessionManagerHelper.createDimensionFromStorageString(storageString);
        Dimension expected = new Dimension(317, 115);
        assertEquals(expected, actual);
    }

    @Test
    public void createDimensionFromGarbageStorageString() {
        String storageString = "murks#1#1";
        Dimension actual = SwingSessionManagerHelper.createDimensionFromStorageString(storageString);
        Dimension expected = SwingSessionManagerHelper.NOT_INITIALISED_DIMENSION;
        assertEquals(expected, actual);
    }

    @Test
    public void createColorStorageStringWhite() {
        Color color = Color.WHITE;
        String actual = SwingSessionManagerHelper.createColorStorageString(color);
        String expected = "255#255#255#255";
        assertEquals(expected, actual);
    }

    @Test
    public void createColorStorageStringBlack() {
        Color color = Color.BLACK;
        String actual = SwingSessionManagerHelper.createColorStorageString(color);
        String expected = "0#0#0#255";
        assertEquals(expected, actual);
    }

    @Test
    public void createColorStorageString() {
        Color color = new Color(217, 115, 105);
        String actual = SwingSessionManagerHelper.createColorStorageString(color);
        String expected = "217#115#105#255";
        assertEquals(expected, actual);
    }

    @Test
    public void createColorFromStorageStringWhite() {
        String storageString = "255#255#255#255";
        Color actual = SwingSessionManagerHelper.createColorFromStorageString(storageString);
        Color expected = Color.WHITE;
        assertEquals(expected, actual);
    }

    @Test
    public void createColorFromStorageStringBlack() {
        String storageString = "0#0#0#255";
        Color actual = SwingSessionManagerHelper.createColorFromStorageString(storageString);
        Color expected = Color.BLACK;
        assertEquals(expected, actual);
    }

    @Test
    public void createColorFromStorageString() {
        String storageString = "217#115#105#255";
        Color actual = SwingSessionManagerHelper.createColorFromStorageString(storageString);
        Color expected = new Color(217, 115, 105, 255);
        assertEquals(expected, actual);
    }

    @Test
    public void createColorFromGarbageStorageString() {
        String storageString = "17#115.7#131#24";
        Color actual = SwingSessionManagerHelper.createColorFromStorageString(storageString);
        Color expected = Color.WHITE;
        assertEquals(expected, actual);
    }

    @Test
    public void createColorsStorageString() {
        List<Color> colors = CollectionsHelper.buildListFrom(Color.WHITE, Color.BLACK,
                new Color(217, 115, 105, 255));
        String actual = SwingSessionManagerHelper.createColorsStorageString(colors);
        String expected = "255#255#255#255"
                + "#-#" + "0#0#0#255"
                + "#-#" + "217#115#105#255"
                ;
        assertEquals(expected, actual);
    }

    @Test
    public void createColorsListFromStorageString() {
        String storageString = "255#255#255#255"
                + "#-#" + "0#0#0#255"
                + "#-#" + "217#115#105#255"
                ;
        List<Color> actualColors =
                SwingSessionManagerHelper.createColorsListFromStorageString(storageString);
        List<Color> expectedColors = CollectionsHelper.buildListFrom(Color.WHITE, Color.BLACK,
                new Color(217, 115, 105, 255));
        assertEquals(expectedColors, actualColors);
    }

}
