package de.duehl.vocabulary.japanese.common.persistence;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import de.duehl.basics.io.FileHelper;
import de.duehl.basics.persistence.SessionManagerHelper;
import de.duehl.basics.text.NumberString;
import de.duehl.basics.text.handling.BooleanHandling;
import de.duehl.basics.version.Version;
import de.duehl.swing.persistance.SwingSessionManagerHelper;
import de.duehl.vocabulary.japanese.common.color.data.ColorModificationType;
import de.duehl.vocabulary.japanese.common.data.TranslationDirection;
import de.duehl.vocabulary.japanese.common.data.VocablesShuffleType;
import de.duehl.vocabulary.japanese.common.data.VocablesSubsetSelectionType;
import de.duehl.vocabulary.japanese.common.data.VocabularySortOrder;
import de.duehl.vocabulary.japanese.common.persistence.data.EmptyTranslationsAcceptance;
import de.duehl.vocabulary.japanese.common.persistence.data.HistoricalOwnListPersistanceDataList;

/**
 * Dieses Klasse regelt den Umgang mit persistenten Informationen, die zwischen zwei Läufen des
 * Programms bewahrt werden sollen. Sie stellt Methoden bereit, diese Daten zu speichern und zu
 * lesen.
 *
 * @version 1.01     2026-01-19
 * @author Christian Dühl
 */

public class SessionManager {

    private static final String STRING_SELECTION_FONT_SIZE = "options.stringSelectionFontSize";
    private static final String STRING_SELECTION_LABEL_FONT_SIZE =
            "options.stringSelectionLabelFontSize";
    private static final String LAST_USED_VERSION = "options.lastUsedVersion";
    private static final String VOCABULARIES_PATH = "options.vocabulariesPath";
    private static final String SWITCH_TO_NEXT_VOCABLE =
            "options.switchToNextVocableAfterEntringTranslation";
    private static final String SHOW_ROMAJI = "options.showRomajiWhenTestingVocable";
    private static final String SHOW_PRONUNCIATION = "options.showPronunciationWhenTestingVocable";
    private static final String SHOW_COMMENT = "options.showCommentWhenTestingVocable";
    private static final String SHOW_VOCABULARY_DESCRIPTION =
            "options.showVocabularyDescriptionWhenTestingVocable";
    private static final String BACKGROUND_COLORING_TESTING =
            "options.colorVocableDependingOnLastSuccessWhenTestingVocable";
    private static final String BACKGROUND_COLORING_VIEWING =
            "options.colorVocableDependingOnLastSuccessWhenViewingVocable";
    private static final String VOCABULARY_SORT_ORDER = "options.vocabularySortOrder";
    private static final String FOREGROUND_COLORING_VOCABULARY =
            "options.colorVocabularyDependingOnLastSuccess";
    private static final String VOCABLES_SHUFFLE_TYPE = "options.vocablesShuffleType";
    private static final String VOCABLES_SUBSET_SELECTION_TYPE =
            "options.vocablesSubsetSelectionType";
    private static final String REVERSE_SHUFFLED_VOCABLES = "options.reverseShuffledVocables";
    private static final String USE_VOCABLES_SHUFFLE_TYPE_FOR_LISTS =
            "options.useVocablesShuffleTypeForShowingListsAndSheetsToo";
    private static final String HIDE_START_OF_VOCABULARY =
            "options.hideStartOfVocabularyDescription";
    private static final String USE_TEST_ORDER_FOR_VIEWING =
            "options.useTestOrderForViewingVocables";
    private static final String LAST_SHOWN_MAIN_TAB_INDEX = "options.lastShownMainTabIndex";
    private static final String LAST_SHOWN_VOCABULARY_CATEGORY_TAB_INDEX =
            "options.lastShownVocabularyCategoryTabIndex";
    private static final String LAST_SHOWN_OWN_LIST_CATEGORY_TAB_INDEX =
            "options.lastShownOwnListCategoryTabIndex";
    private static final String LAST_SHOWN_VOCABULARY_SUB_CATEGORY_TAB_INDEX =
            "options.lastShownVocabularySubCategoryTabIndex";
    private static final String LAST_SHOWN_OWN_LIST_SUB_CATEGORY_TAB_INDEX =
            "options.lastShownOwnListSubCategoryTabIndex";
    private static final String SHOW_SUCCESS_PERCENT_IN_VOCABULARY_BAR =
            "options.showSuccessPercentInVocabularyBar";
    private static final String NUMBER_OF_VOCABULARY_BAR_COLUMNS =
            "options.numberOfVocabularyBarColumns";
    private static final String LAST_USED_OWN_LIST_DIRECTORY =
            "options.lastUsedOwnListImportExportDirectory";
    private static final String NUMBER_OF_DATASETS_PER_PAGE_OLE =
            "options.numberOfDatasetsPerPageOfOwnListEditor";
    private static final String NUMBER_OF_COLUMNS_OLE = "options.numberOfColumnsOfOwnListEditor";
    private static final String LOCATION_OF_OLE = "options.locationOfOwnListEditor";
    private static final String SIZE_OF_OLE = "options.sizeOfOwnListEditor";
    private static final String HIDE_VOCABLES_FROM_LIST_OLE =
            "options.hideVocablesFromListInSearchAreaInOwnListEditor";
    private static final String ALLOW_MISSING_MP3 =
            "options.allowMissingMp3";
    private static final String REPORT_MISSING_MP3 =
            "options.reportMissingMp3";
    private static final String NUMBER_OF_COLUMNS_OF_BLATT =
            "options.numberOfColumnsOfBlattDarstellung";
    private static final String INFORM_ABOUT_DOUBLE_MP3 = "options.informAboutDoubleMp3AtStartup";
    private static final String ONLY_USE_FIRST_GERMAN_TERM_TESTING =
            "options.useOnlyFirstGermanTermWhenTestingVocableGermanToJapanese";
    private static final String TRANSLATION_DIRECTION = "options.translationDirection";
    private static final String SHOW_KANJI = "options.showKanjiWhenTestingVocable";
    private static final String HIDE_KANJI_IF_HIRAGANA =
            "options.hideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana";
    private static final String HIDE_ROMAJI_IF_HIRAGANA =
            "options.hideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana";
    private static final String HIDE_PRONUNCIATION_IF_HIRAGANA =
            "options.hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana";
    private static final String SHOW_DETAILS_BUTTON_EINZELDARSTELLUNG =
            "options.showDetailsButtonInEinzeldarstellung";
    private static final String SHOW_DETAILS_BUTTON_VOKABELBLATTDARSTELLUNG =
            "options.showDetailsButtonInVokabelblattDarstellung";
    private static final String HIDE_LESS_START_OF_VOCABULARY_DESCRIPTION_VHS =
            "options.hideLessStartOfVocabularyDescriptionForVhs";
    private static final String INFORM_ABOUT_EQUAL_KANA_AND_KANJI =
            "options.informAboutEqualKanaAndKanjiAtStartup";
    private static final String SHOW_KANA = "options.showKanaWhenTestingVocable";
    private static final String CHECK_NEW_LISTS = "options.checkNewOwnListsVersionAtStart";
    private static final String KANA_IN_KANJI_SIZE_WHEN_TESTING =
            "options.showKanaInKanjiSizeWhenTestingVocable";
    private static final String EMTPY_TRANSLATION_ACCEPTANCE =
            "options.emptyTranslationsAcceptance";
    private static final String NUMBER_OF_DAYS_FOR_A_WHILE = "options.numberOfDaysForAWhile";
    private static final String NUMBER_OF_DAYS_A_VOCABLE_IS_NEW =
            "options.numberOfDaysAVocableIsNew";
    private static final String COLORS = "options.colors";
    private static final String DELTA_FOREGROUND = "options.deltaForForegroundColor";
    private static final String DELTA_BACKGROUND = "options.deltaForBackgroundColor";
    private static final String COLOR_MODIFICATION_TYPE = "options.colorModificationType";
    private static final String USE_SPLIT_PANE_IN_OLE = "options.useSplitPaneInOwnListEditor";
    private static final String VOCABLE_COLOR_STANDARD = "options.vocableColorStandard";
    private static final String VOCABLE_COLOR_KANJI = "options.vocableColorKanji";
    private static final String VOCABLE_COLOR_KANA = "options.vocableColorKana";
    private static final String VOCABLE_COLOR_ROMAJI = "options.vocableColorRomaji";
    private static final String VOCABLE_COLOR_PRONUNCIATION = "options.vocableColorPronunciation";
    private static final String VOCABLE_COLOR_TRANSLATION = "options.vocableColorTranslation";
    private static final String VOCABLE_COLOR_COMMENT = "options.vocableColorComment";
    private static final String VOCABLE_COLOR_SEARCH_WORDS = "options.vocableColorSearchWords";
    private static final String VOCABLE_COLOR_PART_OF_SPEACH = "options.vocableColorPartOfSpeach";
    private static final String SHOW_STARTUP_TIMESTAMP = "options.showTimestampsInStartup";
    private static final String CREATE_MAP_D_TO_J_AT_STARTUP =
            "options.createGermanJapaneseTranslationAtStartup";
    private static final String LAST_USED_7ZIP_DIR = "options.lastUsed7ZipDirectory";
    private static final String LAST_SELECTED_KANJI_SET_TEXT =
            "options.lastSelectedKanjiSetTextInCombobox";
    private static final String INFORM_ABOUT_E_OR_U_IN_AUSSPRACHE =
            "options.informAboutEOrUInVocableFieldAusspracheAtStartup";
    private static final String NUMBER_OF_REMEMBERED_OWN_LISTS =
            "options.numberOfRememberedOwnLists";
    private static final String HISTORICAL_LEFT_OWN_LISTS_IN_LIST_COMMANDER =
            "options.historicalLeftOwnListsInListCommander";
    private static final String HISTORICAL_RIGHT_OWN_LISTS_IN_LIST_COMMANDER =
            "options.historicalRightOwnListsInListCommander";
    private static final String HISTORICAL_OWN_LISTS = "options.historicalOwnLists";
    private static final String SHOW_ALL_TRANSLATIONS = "options.showAllTranslations";
    private static final String NUMBER_OF_SHOWN_TRANSLATIONS = "options.numberOfShownTranslations";
    private static final String SHOW_NUMBER_OF_LISTS = "options.showNumberOfListsInLister";
    private static final String SHOW_BUTTONS_IN_EINZELDARSTELLUNG =
            "options.showDetailBottomButonsInEinzeldarstellungToo";
    private static final String SHOW_BUTTONS_IN_TRANSLATION_EVALUATION =
            "options.showDetailBottomButonsInTranslationEvaluationToo";
    private static final String PLAY_MP3_WHEN_TESTING = "options.playMP3WhenTestingVocables";
    private static final String COLOR_USER_INPUT_WHEN_TESTING_KANJI =
            "options.colorUserInputAtKanjiTestEvaluationDialog";
    private static final String STOP_AFTER_STARTUP = "options.stopAfterStartup";
    private static final String WAIT_FOR_X_AFTER_STARTUP = "options.waitForXAfterStartup";
    private static final String GT_SHOWN_KANJI = "options.grammarTableShowKanji";
    private static final String GT_SHOW_KANA = "options.grammarTableShowKana";
    private static final String GT_SHOW_ROMAJI = "options.grammarTableShowRomaji";
    private static final String GT_SEARCH_BASE_FORMS = "options.grammarTableSearchForBaseForms";
    private static final String GT_LEAVE_BLANK = "options.grammarTableLeaveNotFoundFormsBlank";
    private static final String GT_VERB_HEADERS = "options.grammarTableVerbHeaders";
    private static final String GT_I_ADJECTIVE_HEADERS = "options.grammarTableIAdjectiveHeaders";
    private static final String GT_NA_ADJECTIVE_HEADERS = "options.grammarTableNaAdjectiveHeaders";
    private static final String GT_USE_USER_HEADERS = "options.grammarTableUseUserHeaders";
    private static final String GT_USER_HEADER_SAVE_LOAD_DIR =
            "options.grammerTableLastUsedOwnHeadersDirectory";
    private static final String CLOSE_SINGLE_VIEW_WHEH_SHOWING_DETAILS =
            "options.closeSingleViewWhenOpeningDetailsAfterSimpleSearch";
    private static final String SHOW_WARNUNGS_AFTER_START = "options.showWarningsAfterStart";
    private static final String SHOW_KANJI_TEST_AT_PARAMETER_POSITION =
            "options.showKanjiTestAtParameterDialogPosition";
    private static final String SHOW_KANJI_NUMBER_IN_NOTENOOK =
            "options.showKanjiNumberInMyHandwrittenKanjiNotebooks";
    private static final String NUMBER_OF_CHARS_IN_BAR_TITLE =
            "options.numberOfCharsInVocableBarTitleShown";

    /** Name des Verzeichnisses mit den Einstellungen ohne Pfad. */
    private static final String VOCABLE_TRAINER_DIRECTORY_IN_HOME_DIR = ".JapaneseVocableTrainer";

    /** Name des Verzeichnisses mit den Einstellungen mit Pfad. */
    public static final String VOCABLE_TRAINER_DIRECTORY =
            FileHelper.getAndPerhapsCreateDirectoryAtHome(VOCABLE_TRAINER_DIRECTORY_IN_HOME_DIR);


    /** Der Name des Sessions-Files ohne Pfad. */
    private static final String SESSION_BARE_FILENAME = "japanese_vocable_trainer.xml";

    /** Der Name des Sessions-Files mit Pfad. */
    public static final String SESSION_FILENAME = FileHelper.concatPathes(
            VOCABLE_TRAINER_DIRECTORY, SESSION_BARE_FILENAME);


    /**
     * Die Version des Vokabel-Trainers (erhältlich mit
     * new VocabularyTrainerVersion().getVersion()).
     */
    private final Version vocabularyTrainerVersion;

    /** Ein Options-Objekt, dort werden die Optionen des Programms verwaltet. */
    private final Options options;

    /** Objekt, das sich um das Speichern und Lesen der Property-Datei kümmert. */
    private final SessionManagerHelper helper;

    /**
     * Konstruktor.
     *
     * @param vocabularyTrainerVersion
     *            Die Version des Vokabel-Trainers (erhältlich mit
     *            new VocabularyTrainerVersion().getVersion()).
     */
    public SessionManager(Version vocabularyTrainerVersion) {
        this(vocabularyTrainerVersion, new Options(vocabularyTrainerVersion));
    }

    /**
     * Konstruktor.
     *
     * @param vocabularyTrainerVersion
     *            Die Version des Vokabel-Trainers (erhältlich mit new
     *            VocabularyTrainerVersion().getVersion()).
     * @param options
     *            Das Options-Objekt, dort werden die Optionen des Programms verwaltet.
     */
    public SessionManager(Version vocabularyTrainerVersion, Options options) {
        this.vocabularyTrainerVersion = vocabularyTrainerVersion;
        this.options = options;

        helper = new SessionManagerHelper(SESSION_FILENAME);
    }

    /** Lädt die Optionen, falls vorhanden. */
    public Options load() {
        loadInternal();
        return options;
    }

    private void loadInternal() {
        Properties properties = helper.readProperties();
        if (null != properties) {
            loadOptions(properties);
            adjustHistoricalOwnListsInListCommander();
        }
    }

    /** Lädt die Optionen. Hier werden die eigentlichen Daten aus den Properties gelesen. */
    private void loadOptions(Properties properties) {
        if (properties.containsKey(STRING_SELECTION_FONT_SIZE)) {
            String stringSelectionFontSizeAsString =
                    properties.getProperty(STRING_SELECTION_FONT_SIZE);
            int stringSelectionFontSize =
                    NumberString.parseIntIgnore(stringSelectionFontSizeAsString, -1);
            options.setStringSelectionFontSize(stringSelectionFontSize);
        }
        if (properties.containsKey(STRING_SELECTION_LABEL_FONT_SIZE)) {
            String stringSelectionLabelFontSizeAsString =
                    properties.getProperty(STRING_SELECTION_LABEL_FONT_SIZE);
            int stringSelectionLabelFontSize =
                    NumberString.parseIntIgnore(stringSelectionLabelFontSizeAsString, -1);
            options.setStringSelectionLabelFontSize(stringSelectionLabelFontSize);
        }
        if (properties.containsKey(LAST_USED_VERSION)) {
            String serializedVersion = properties.getProperty(LAST_USED_VERSION);
            Version lastUsedVersion = Version.createFromSerializedString(serializedVersion);
            options.setLastUsedVersion(lastUsedVersion);
        }
        if (properties.containsKey(VOCABULARIES_PATH)) {
            String vocabulariesPath = properties.getProperty(VOCABULARIES_PATH);
            options.setVocabulariesPath(vocabulariesPath);
        }
        if (properties.containsKey(SWITCH_TO_NEXT_VOCABLE)) {
            String switchToNextVocableAfterEntringTranslationAsString =
                    properties.getProperty(SWITCH_TO_NEXT_VOCABLE);
            boolean switchToNextVocableAfterEntringTranslation =
                    BooleanHandling.stringToBoolean(
                            switchToNextVocableAfterEntringTranslationAsString);
            options.setSwitchToNextVocableAfterEntringTranslation(
                    switchToNextVocableAfterEntringTranslation);
        }
        if (properties.containsKey(SHOW_ROMAJI)) {
            String showRomajiWhenTestingVocableAsString = properties.getProperty(SHOW_ROMAJI);
            boolean showRomajiWhenTestingVocable = BooleanHandling.stringToBoolean(
                    showRomajiWhenTestingVocableAsString);
            options.setShowRomajiWhenTestingVocable(showRomajiWhenTestingVocable);
        }
        if (properties.containsKey(SHOW_PRONUNCIATION)) {
            String showPronunciationWhenTestingVocableAsString =
                    properties.getProperty(SHOW_PRONUNCIATION);
            boolean showPronunciationWhenTestingVocable = BooleanHandling.stringToBoolean(
                    showPronunciationWhenTestingVocableAsString);
            options.setShowPronunciationWhenTestingVocable(showPronunciationWhenTestingVocable);
        }
        if (properties.containsKey(SHOW_COMMENT)) {
            String showCommentWhenTestingVocableAsString = properties.getProperty(SHOW_COMMENT);
            boolean showCommentWhenTestingVocable = BooleanHandling.stringToBoolean(
                    showCommentWhenTestingVocableAsString);
            options.setShowCommentWhenTestingVocable(showCommentWhenTestingVocable);
        }
        if (properties.containsKey(SHOW_VOCABULARY_DESCRIPTION)) {
            String showVocabularyDescriptionWhenTestingVocableAsString =
                    properties.getProperty(SHOW_VOCABULARY_DESCRIPTION);
            boolean showVocabularyDescriptionWhenTestingVocable = BooleanHandling.stringToBoolean(
                    showVocabularyDescriptionWhenTestingVocableAsString);
            options.setShowVocabularyDescriptionWhenTestingVocable(
                    showVocabularyDescriptionWhenTestingVocable);
        }
        if (properties.containsKey(BACKGROUND_COLORING_TESTING)) {
            String colorVocableDependingOnLastSuccessWhenTestingVocableAsString =
                    properties.getProperty(BACKGROUND_COLORING_TESTING);
            boolean colorVocableDependingOnLastSuccessWhenTestingVocable =
                    BooleanHandling.stringToBoolean(
                            colorVocableDependingOnLastSuccessWhenTestingVocableAsString);
            options.setColorVocableDependingOnLastSuccessWhenTestingVocable(
                    colorVocableDependingOnLastSuccessWhenTestingVocable);
        }
        if (properties.containsKey(BACKGROUND_COLORING_VIEWING)) {
            String colorVocableDependingOnLastSuccessWhenViewingVocableAsString =
                    properties.getProperty(BACKGROUND_COLORING_VIEWING);
            boolean colorVocableDependingOnLastSuccessWhenViewingVocable =
                    BooleanHandling.stringToBoolean(
                            colorVocableDependingOnLastSuccessWhenViewingVocableAsString);
            options.setColorVocableDependingOnLastSuccessWhenViewingVocable(
                    colorVocableDependingOnLastSuccessWhenViewingVocable);
        }
        if (properties.containsKey(VOCABULARY_SORT_ORDER)) {
            String vocabularySortOrderAsString = properties.getProperty(VOCABULARY_SORT_ORDER);
            VocabularySortOrder vocabularySortOrder = VocabularySortOrder.determineByName(
                    vocabularySortOrderAsString);
            options.setVocabularySortOrder(vocabularySortOrder);
        }
        if (properties.containsKey(FOREGROUND_COLORING_VOCABULARY)) {
            String colorVocabularyDependingOnLastSuccessAsString =
                    properties.getProperty(FOREGROUND_COLORING_VOCABULARY);
            boolean colorVocabularyDependingOnLastSuccess =
                    BooleanHandling.stringToBoolean(colorVocabularyDependingOnLastSuccessAsString);
            options.setColorVocabularyDependingOnLastSuccess(colorVocabularyDependingOnLastSuccess);
        }
        if (properties.containsKey(VOCABLES_SHUFFLE_TYPE)) {
            String vocablesShuffleTypeName = properties.getProperty(VOCABLES_SHUFFLE_TYPE);
            VocablesShuffleType vocablesShuffleType = VocablesShuffleType.determineByName(
                    vocablesShuffleTypeName);
            options.setVocablesShuffleType(vocablesShuffleType);
        }
        if (properties.containsKey(VOCABLES_SUBSET_SELECTION_TYPE)) {
            String vocablesSubsetSelectionTypeName =
                    properties.getProperty(VOCABLES_SUBSET_SELECTION_TYPE);
            VocablesSubsetSelectionType vocablesSubsetSelectionType =
                    VocablesSubsetSelectionType.determineByName(vocablesSubsetSelectionTypeName);
            options.setVocablesSubsetSelectionType(vocablesSubsetSelectionType);
        }
        if (properties.containsKey(REVERSE_SHUFFLED_VOCABLES)) {
            String reverseShuffledVocablesAsString =
                    properties.getProperty(REVERSE_SHUFFLED_VOCABLES);
            boolean reverseShuffledVocables =
                    BooleanHandling.stringToBoolean(reverseShuffledVocablesAsString);
            options.setReverseShuffledVocables(reverseShuffledVocables);
        }
        if (properties.containsKey(USE_VOCABLES_SHUFFLE_TYPE_FOR_LISTS)) {
            String useVocablesShuffleTypeForShowingListsAndSheetsTooAsString =
                    properties.getProperty(USE_VOCABLES_SHUFFLE_TYPE_FOR_LISTS);
            boolean useVocablesShuffleTypeForShowingListsAndSheetsToo =
                    BooleanHandling.stringToBoolean(
                            useVocablesShuffleTypeForShowingListsAndSheetsTooAsString);
            options.setUseVocablesShuffleTypeForShowingListsAndSheetsToo(
                    useVocablesShuffleTypeForShowingListsAndSheetsToo);
        }
        if (properties.containsKey(HIDE_START_OF_VOCABULARY)) {
            String hideStartOfVocabularyDescriptionAsString =
                    properties.getProperty(HIDE_START_OF_VOCABULARY);
            boolean hideStartOfVocabularyDescription = BooleanHandling.stringToBoolean(
                    hideStartOfVocabularyDescriptionAsString);
            options.setHideStartOfVocabularyDescription(hideStartOfVocabularyDescription);
        }
        if (properties.containsKey(USE_TEST_ORDER_FOR_VIEWING)) {
            String useTestOrderForViewingVocablesAsString =
                    properties.getProperty(USE_TEST_ORDER_FOR_VIEWING);
            boolean useTestOrderForViewingVocables = BooleanHandling.stringToBoolean(
                    useTestOrderForViewingVocablesAsString);
            options.setUseTestOrderForViewingVocables(useTestOrderForViewingVocables);
        }
        if (properties.containsKey(LAST_SHOWN_MAIN_TAB_INDEX)) {
            String lastShownMainTabIndexAsString = properties.getProperty(LAST_SHOWN_MAIN_TAB_INDEX);
            int lastShownMainTabIndex = NumberString.parseIntIgnore(lastShownMainTabIndexAsString, -1);
            options.setLastShownMainTabIndex(lastShownMainTabIndex);
        }
        if (properties.containsKey(LAST_SHOWN_VOCABULARY_CATEGORY_TAB_INDEX)) {
            String lastShownVocabularyCategoryTabIndexAsString =
                    properties.getProperty(LAST_SHOWN_VOCABULARY_CATEGORY_TAB_INDEX);
            int lastShownVocabularyCategoryTabIndex =
                    NumberString.parseIntIgnore(lastShownVocabularyCategoryTabIndexAsString, -1);
            options.setLastShownVocabularyCategoryTabIndex(lastShownVocabularyCategoryTabIndex);
        }
        if (properties.containsKey(LAST_SHOWN_OWN_LIST_CATEGORY_TAB_INDEX)) {
            String lastShownOwnListCategoryTabIndexAsString =
                    properties.getProperty(LAST_SHOWN_OWN_LIST_CATEGORY_TAB_INDEX);
            int lastShownOwnListCategoryTabIndex =
                    NumberString.parseIntIgnore(lastShownOwnListCategoryTabIndexAsString, -1);
            options.setLastShownOwnListCategoryTabIndex(lastShownOwnListCategoryTabIndex);
        }
        if (properties.containsKey(LAST_SHOWN_VOCABULARY_SUB_CATEGORY_TAB_INDEX)) {
            String lastShownVocabularySubCategoryTabIndexAsString =
                    properties.getProperty(LAST_SHOWN_VOCABULARY_SUB_CATEGORY_TAB_INDEX);
            int lastShownVocabularySubCategoryTabIndex =
                    NumberString.parseIntIgnore(lastShownVocabularySubCategoryTabIndexAsString, -1);
            options.setLastShownVocabularySubCategoryTabIndex(
                    lastShownVocabularySubCategoryTabIndex);
        }
        if (properties.containsKey(LAST_SHOWN_OWN_LIST_SUB_CATEGORY_TAB_INDEX)) {
            String lastShownOwnListSubCategoryTabIndexAsString =
                    properties.getProperty(LAST_SHOWN_OWN_LIST_SUB_CATEGORY_TAB_INDEX);
            int lastShownOwnListSubCategoryTabIndex =
                    NumberString.parseIntIgnore(lastShownOwnListSubCategoryTabIndexAsString, -1);
            options.setLastShownOwnListSubCategoryTabIndex(lastShownOwnListSubCategoryTabIndex);
        }
        if (properties.containsKey(SHOW_SUCCESS_PERCENT_IN_VOCABULARY_BAR)) {
            String showSuccessPercentInVocabularyBarAsString =
                    properties.getProperty(SHOW_SUCCESS_PERCENT_IN_VOCABULARY_BAR);
            boolean showSuccessPercentInVocabularyBar = BooleanHandling.stringToBoolean(
                    showSuccessPercentInVocabularyBarAsString);
            options.setShowSuccessPercentInVocabularyBar(showSuccessPercentInVocabularyBar);
        }
        if (properties.containsKey(NUMBER_OF_VOCABULARY_BAR_COLUMNS)) {
            String numberOfVocabularyBarColumnsAsString =
                    properties.getProperty(NUMBER_OF_VOCABULARY_BAR_COLUMNS);
            int numberOfVocabularyBarColumns = NumberString.parseIntIgnore(
                    numberOfVocabularyBarColumnsAsString, -1);
            options.setNumberOfVocabularyBarColumns(numberOfVocabularyBarColumns);
        }
        if (properties.containsKey(LAST_USED_OWN_LIST_DIRECTORY)) {
            String lastUsedOwnListImportExportDirectory =
                    properties.getProperty(LAST_USED_OWN_LIST_DIRECTORY);
            options.setLastUsedOwnListImportExportDirectory(lastUsedOwnListImportExportDirectory);
        }
        if (properties.containsKey(NUMBER_OF_DATASETS_PER_PAGE_OLE)) {
            String numberOfDatasetsPerPageOfOwnListEditorAsString =
                    properties.getProperty(NUMBER_OF_DATASETS_PER_PAGE_OLE);
            int numberOfDatasetsPerPageOfOwnListEditor = NumberString.parseIntIgnore(
                    numberOfDatasetsPerPageOfOwnListEditorAsString, -1);
            options.setNumberOfDatasetsPerPageOfOwnListEditor(
                    numberOfDatasetsPerPageOfOwnListEditor);
        }
        if (properties.containsKey(NUMBER_OF_COLUMNS_OLE)) {
            String numberOfColumnsOfOwnListEditorAsString =
                    properties.getProperty(NUMBER_OF_COLUMNS_OLE);
            int numberOfColumnsOfOwnListEditor = NumberString.parseIntIgnore(
                    numberOfColumnsOfOwnListEditorAsString, -1);
            options.setNumberOfColumnsOfOwnListEditor(numberOfColumnsOfOwnListEditor);
        }
        if (properties.containsKey(LOCATION_OF_OLE)) {
            String locationOfOwnListEditorAsString = properties.getProperty(LOCATION_OF_OLE);
            Point locationOfOwnListEditor = SwingSessionManagerHelper.createPointFromStorageString(
                    locationOfOwnListEditorAsString);
            options.setLocationOfOwnListEditor(locationOfOwnListEditor);
        }
        if (properties.containsKey(SIZE_OF_OLE)) {
            String sizeOfOwnListEditorAsString = properties.getProperty(SIZE_OF_OLE);
            Dimension sizeOfOwnListEditor =
                    SwingSessionManagerHelper.createDimensionFromStorageString(
                            sizeOfOwnListEditorAsString);
            options.setSizeOfOwnListEditor(sizeOfOwnListEditor);
        }
        if (properties.containsKey(HIDE_VOCABLES_FROM_LIST_OLE)) {
            String hideVocablesFromListInSearchAreaInOwnListEditorAsString =
                    properties.getProperty(HIDE_VOCABLES_FROM_LIST_OLE);
            boolean hideVocablesFromListInSearchAreaInOwnListEditor =
                    BooleanHandling.stringToBoolean(
                            hideVocablesFromListInSearchAreaInOwnListEditorAsString);
            options.setHideVocablesFromListInSearchAreaInOwnListEditor(
                    hideVocablesFromListInSearchAreaInOwnListEditor);
        }
        if (properties.containsKey(ALLOW_MISSING_MP3)) {
            String allowMissingMp3AsString =
                    properties.getProperty(ALLOW_MISSING_MP3);
            boolean allowMissingMp3 = BooleanHandling.stringToBoolean(allowMissingMp3AsString);
            options.setAllowMissingMp3(allowMissingMp3);
        }
        if (properties.containsKey(REPORT_MISSING_MP3)) {
            String allowReportMp3AsString =
                    properties.getProperty(REPORT_MISSING_MP3);
            boolean reportMissingMp3 = BooleanHandling.stringToBoolean(allowReportMp3AsString);
            options.setReportMissingMp3(reportMissingMp3);
        }
        if (properties.containsKey(NUMBER_OF_COLUMNS_OF_BLATT)) {
            String numberOfColumnsOfBlattDarstellungAsString =
                    properties.getProperty(NUMBER_OF_COLUMNS_OF_BLATT);
            int numberOfColumnsOfBlattDarstellung = NumberString.parseIntIgnore(
                    numberOfColumnsOfBlattDarstellungAsString, -1);
            options.setNumberOfColumnsOfBlattDarstellung(numberOfColumnsOfBlattDarstellung);
        }
        if (properties.containsKey(INFORM_ABOUT_DOUBLE_MP3)) {
            String informAboutDoubleMp3AtStartupAsString =
                    properties.getProperty(INFORM_ABOUT_DOUBLE_MP3);
            boolean informAboutDoubleMp3AtStartup =
                    BooleanHandling.stringToBoolean(informAboutDoubleMp3AtStartupAsString);
            options.setInformAboutDoubleMp3AtStartup(informAboutDoubleMp3AtStartup);
        }
        if (properties.containsKey(ONLY_USE_FIRST_GERMAN_TERM_TESTING)) {
            String useOnlyFirstGermanTermWhenTestingVocableGermanToJapaneseAsString =
                    properties.getProperty(ONLY_USE_FIRST_GERMAN_TERM_TESTING);
            boolean useOnlyFirstGermanTermWhenTestingVocableGermanToJapanese =
                    BooleanHandling.stringToBoolean(
                            useOnlyFirstGermanTermWhenTestingVocableGermanToJapaneseAsString);
            options.setUseOnlyFirstGermanTermWhenTestingVocableGermanToJapanese(
                    useOnlyFirstGermanTermWhenTestingVocableGermanToJapanese);
        }
        if (properties.containsKey(TRANSLATION_DIRECTION)) {
            String translationDirectionName = properties.getProperty(TRANSLATION_DIRECTION);
            TranslationDirection translationDirection = TranslationDirection.determineByName(
                    translationDirectionName);
            options.setTranslationDirection(translationDirection);
        }
        if (properties.containsKey(SHOW_KANJI)) {
            String showKanjiWhenTestingVocableAsString = properties.getProperty(SHOW_KANJI);
            boolean showKanjiWhenTestingVocable = BooleanHandling.stringToBoolean(
                    showKanjiWhenTestingVocableAsString);
            options.setShowKanjiWhenTestingVocable(showKanjiWhenTestingVocable);
        }
        if (properties.containsKey(HIDE_KANJI_IF_HIRAGANA)) {
            String hideKanjiWhenTestingVocableAndKanaContainsOnlyHiraganaAsString =
                    properties.getProperty(HIDE_KANJI_IF_HIRAGANA);
            boolean hideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana =
                    BooleanHandling.stringToBoolean(
                            hideKanjiWhenTestingVocableAndKanaContainsOnlyHiraganaAsString);
            options.setHideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana(
                    hideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana);
        }
        if (properties.containsKey(HIDE_ROMAJI_IF_HIRAGANA)) {
            String hideRomajiWhenTestingVocableAndKanaContainsOnlyHiraganaAsString =
                    properties.getProperty(HIDE_ROMAJI_IF_HIRAGANA);
            boolean hideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana =
                    BooleanHandling.stringToBoolean(
                            hideRomajiWhenTestingVocableAndKanaContainsOnlyHiraganaAsString);
            options.setHideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana(
                    hideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana);
        }
        if (properties.containsKey(HIDE_PRONUNCIATION_IF_HIRAGANA)) {
            String hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiraganaAsString =
                    properties.getProperty(HIDE_PRONUNCIATION_IF_HIRAGANA);
            boolean hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana =
                    BooleanHandling.stringToBoolean(
                            hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiraganaAsString);
            options.setHidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana(
                    hidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana);
        }
        if (properties.containsKey(SHOW_DETAILS_BUTTON_EINZELDARSTELLUNG)) {
            String showDetailsButtonInEinzeldarstellungAsString =
                    properties.getProperty(SHOW_DETAILS_BUTTON_EINZELDARSTELLUNG);
            boolean showDetailsButtonInEinzeldarstellung =
                    BooleanHandling.stringToBoolean(
                            showDetailsButtonInEinzeldarstellungAsString);
            options.setShowDetailsButtonInEinzeldarstellung(showDetailsButtonInEinzeldarstellung);
        }
        if (properties.containsKey(SHOW_DETAILS_BUTTON_VOKABELBLATTDARSTELLUNG)) {
            String showDetailsButtonInVokabelblattDarstellungAsString =
                    properties.getProperty(SHOW_DETAILS_BUTTON_VOKABELBLATTDARSTELLUNG);
            boolean showDetailsButtonInVokabelblattDarstellung =
                    BooleanHandling.stringToBoolean(
                            showDetailsButtonInVokabelblattDarstellungAsString);
            options.setShowDetailsButtonInVokabelblattDarstellung(
                    showDetailsButtonInVokabelblattDarstellung);
        }
        if (properties.containsKey(HIDE_LESS_START_OF_VOCABULARY_DESCRIPTION_VHS)) {
            String hideLessStartOfVocabularyDescriptionForVhsAsString =
                    properties.getProperty(HIDE_LESS_START_OF_VOCABULARY_DESCRIPTION_VHS);
            boolean hideLessStartOfVocabularyDescriptionForVhs =
                    BooleanHandling.stringToBoolean(
                            hideLessStartOfVocabularyDescriptionForVhsAsString);
            options.setHideLessStartOfVocabularyDescriptionForVhs(
                    hideLessStartOfVocabularyDescriptionForVhs);
        }
        if (properties.containsKey(INFORM_ABOUT_EQUAL_KANA_AND_KANJI)) {
            String informAboutEqualKanaAndKanjiAtStartupAsString =
                    properties.getProperty(INFORM_ABOUT_EQUAL_KANA_AND_KANJI);
            boolean informAboutEqualKanaAndKanjiAtStartup =
                    BooleanHandling.stringToBoolean(informAboutEqualKanaAndKanjiAtStartupAsString);
            options.setInformAboutEqualKanaAndKanjiAtStartup(informAboutEqualKanaAndKanjiAtStartup);
        }
        if (properties.containsKey(SHOW_KANA)) {
            String showKanaWhenTestingVocableAsString = properties.getProperty(SHOW_KANA);
            boolean showKanaWhenTestingVocable = BooleanHandling.stringToBoolean(
                    showKanaWhenTestingVocableAsString);
            options.setShowKanaWhenTestingVocable(showKanaWhenTestingVocable);
        }
        if (properties.containsKey(CHECK_NEW_LISTS)) {
            String checkNewOwnListsVersionAtStartAsString = properties.getProperty(CHECK_NEW_LISTS);
            boolean checkNewOwnListsVersionAtStart = BooleanHandling.stringToBoolean(
                    checkNewOwnListsVersionAtStartAsString);
            options.setCheckNewOwnListsVersionAtStart(checkNewOwnListsVersionAtStart);
        }
        if (properties.containsKey(KANA_IN_KANJI_SIZE_WHEN_TESTING)) {
            String showKanaInKanjiSizeWhenTestingVocableAsString =
                    properties.getProperty(KANA_IN_KANJI_SIZE_WHEN_TESTING);
            boolean showKanaInKanjiSizeWhenTestingVocable = BooleanHandling.stringToBoolean(
                    showKanaInKanjiSizeWhenTestingVocableAsString);
            options.setShowKanaInKanjiSizeWhenTestingVocable(showKanaInKanjiSizeWhenTestingVocable);
        }
        if (properties.containsKey(EMTPY_TRANSLATION_ACCEPTANCE)) {
            String emptyTranslationsAcceptanceString =
                    properties.getProperty(EMTPY_TRANSLATION_ACCEPTANCE);
            EmptyTranslationsAcceptance emptyTranslationsAcceptance =
                    EmptyTranslationsAcceptance.createFromPersistence(
                            emptyTranslationsAcceptanceString);
            options.setEmptyTranslationsAcceptance(emptyTranslationsAcceptance);
        }
        if (properties.containsKey(NUMBER_OF_DAYS_FOR_A_WHILE)) {
            String numberOfDaysForAWhileAsString =
                    properties.getProperty(NUMBER_OF_DAYS_FOR_A_WHILE);
            int numberOfDaysForAWhile =
                    NumberString.parseIntIgnore(numberOfDaysForAWhileAsString, -1);
            options.setNumberOfDaysForAWhile(numberOfDaysForAWhile);
        }
        if (properties.containsKey(NUMBER_OF_DAYS_A_VOCABLE_IS_NEW)) {
            String numberOfDaysAVocableIsNewAsString =
                    properties.getProperty(NUMBER_OF_DAYS_A_VOCABLE_IS_NEW);
            int numberOfDaysAVocableIsNew =
                    NumberString.parseIntIgnore(numberOfDaysAVocableIsNewAsString, -1);
            options.setNumberOfDaysAVocableIsNew(numberOfDaysAVocableIsNew);
        }
        if (properties.containsKey(COLORS)) {
            String colorsStorageString = properties.getProperty(COLORS);
            List<Color> colors = SwingSessionManagerHelper.createColorsListFromStorageString(
                    colorsStorageString);
            options.setColors(colors);
        }
        if (properties.containsKey(DELTA_FOREGROUND)) {
            String deltaForForegroundColorAsString = properties.getProperty(DELTA_FOREGROUND);
            int deltaForForegroundColor =
                    NumberString.parseIntIgnore(deltaForForegroundColorAsString, -1);
            options.setDeltaForForegroundColor(deltaForForegroundColor);
        }
        if (properties.containsKey(DELTA_BACKGROUND)) {
            String deltaForBackgroundColorAsString = properties.getProperty(DELTA_BACKGROUND);
            int deltaForBackgroundColor =
                    NumberString.parseIntIgnore(deltaForBackgroundColorAsString, -1);
            options.setDeltaForBackgroundColor(deltaForBackgroundColor);
        }
        if (properties.containsKey(COLOR_MODIFICATION_TYPE)) {
            String colorModificationTypeName = properties.getProperty(COLOR_MODIFICATION_TYPE);
            ColorModificationType colorModificationType = ColorModificationType.determineByName(
                    colorModificationTypeName);
            options.setColorModificationType(colorModificationType);
        }
        if (properties.containsKey(USE_SPLIT_PANE_IN_OLE)) {
            String useSplitPaneInOwnListEditorAsString =
                    properties.getProperty(USE_SPLIT_PANE_IN_OLE);
            boolean useSplitPaneInOwnListEditor = BooleanHandling.stringToBoolean(
                    useSplitPaneInOwnListEditorAsString);
            options.setUseSplitPaneInOwnListEditor(useSplitPaneInOwnListEditor);
        }
        if (properties.containsKey(VOCABLE_COLOR_STANDARD)) {
            String vocableColorStandardString = properties.getProperty(VOCABLE_COLOR_STANDARD);
            Color vocableColorStandard = SwingSessionManagerHelper.createColorFromStorageString(
                    vocableColorStandardString);
            options.setVocableColorStandard(vocableColorStandard);
        }
        if (properties.containsKey(VOCABLE_COLOR_KANJI)) {
            String vocableColorKanjiString = properties.getProperty(VOCABLE_COLOR_KANJI);
            Color vocableColorKanji = SwingSessionManagerHelper.createColorFromStorageString(
                    vocableColorKanjiString);
            options.setVocableColorKanji(vocableColorKanji);
        }
        if (properties.containsKey(VOCABLE_COLOR_KANA)) {
            String vocableColorKanaString = properties.getProperty(VOCABLE_COLOR_KANA);
            Color vocableColorKana = SwingSessionManagerHelper.createColorFromStorageString(
                    vocableColorKanaString);
            options.setVocableColorKana(vocableColorKana);
        }
        if (properties.containsKey(VOCABLE_COLOR_ROMAJI)) {
            String vocableColorRomajiString = properties.getProperty(VOCABLE_COLOR_ROMAJI);
            Color vocableColorRomaji = SwingSessionManagerHelper.createColorFromStorageString(
                    vocableColorRomajiString);
            options.setVocableColorRomaji(vocableColorRomaji);
        }
        if (properties.containsKey(VOCABLE_COLOR_PRONUNCIATION)) {
            String vocableColorPronunciationString = properties.getProperty(
                    VOCABLE_COLOR_PRONUNCIATION);
            Color vocableColorPronunciation = SwingSessionManagerHelper.createColorFromStorageString(
                    vocableColorPronunciationString);
            options.setVocableColorPronunciation(vocableColorPronunciation);
        }
        if (properties.containsKey(VOCABLE_COLOR_TRANSLATION)) {
            String vocableColorTranslationString = properties.getProperty(
                    VOCABLE_COLOR_TRANSLATION);
            Color vocableColorTranslation = SwingSessionManagerHelper.createColorFromStorageString(
                    vocableColorTranslationString);
            options.setVocableColorTranslation(vocableColorTranslation);
        }
        if (properties.containsKey(VOCABLE_COLOR_COMMENT)) {
            String vocableColorCommentString = properties.getProperty(VOCABLE_COLOR_COMMENT);
            Color vocableColorComment = SwingSessionManagerHelper.createColorFromStorageString(
                    vocableColorCommentString);
            options.setVocableColorComment(vocableColorComment);
        }
        if (properties.containsKey(VOCABLE_COLOR_SEARCH_WORDS)) {
            String vocableColorSearchWordsString = properties.getProperty(
                    VOCABLE_COLOR_SEARCH_WORDS);
            Color vocableColorSearchWords = SwingSessionManagerHelper.createColorFromStorageString(
                    vocableColorSearchWordsString);
            options.setVocableColorSearchWords(vocableColorSearchWords);
        }
        if (properties.containsKey(VOCABLE_COLOR_PART_OF_SPEACH)) {
            String vocableColorPartOfSpeachString = properties.getProperty(
                    VOCABLE_COLOR_PART_OF_SPEACH);
            Color vocableColorPartOfSpeach = SwingSessionManagerHelper.createColorFromStorageString(
                    vocableColorPartOfSpeachString);
            options.setVocableColorPartOfSpeach(vocableColorPartOfSpeach);
        }
        if (properties.containsKey(SHOW_STARTUP_TIMESTAMP)) {
            String showTimestampsInStartupAsString = properties.getProperty(SHOW_STARTUP_TIMESTAMP);
            boolean showTimestampsInStartup = BooleanHandling.stringToBoolean(
                    showTimestampsInStartupAsString);
            options.setShowTimestampsInStartup(showTimestampsInStartup);
        }
        if (properties.containsKey(CREATE_MAP_D_TO_J_AT_STARTUP)) {
            String createGermanJapaneseTranslationAtStartupAsString =
                    properties.getProperty(CREATE_MAP_D_TO_J_AT_STARTUP);
            boolean createGermanJapaneseTranslationAtStartup = BooleanHandling.stringToBoolean(
                    createGermanJapaneseTranslationAtStartupAsString);
            options.setCreateGermanJapaneseTranslationAtStartup(
                    createGermanJapaneseTranslationAtStartup);
        }
        if (properties.containsKey(LAST_USED_7ZIP_DIR)) {
            String lastUsed7ZipDirectory = properties.getProperty(LAST_USED_7ZIP_DIR);
            options.setLastUsed7ZipDirectory(lastUsed7ZipDirectory);
        }
        if (properties.containsKey(LAST_SELECTED_KANJI_SET_TEXT)) {
            String lastSelectedKanjiSetTextInCombobox =
                    properties.getProperty(LAST_SELECTED_KANJI_SET_TEXT);
            options.setLastSelectedKanjiSetTextInCombobox(lastSelectedKanjiSetTextInCombobox);
        }
        if (properties.containsKey(INFORM_ABOUT_E_OR_U_IN_AUSSPRACHE)) {
            String informAboutEOrUInVocableFieldAusspracheAtStartupAsString =
                    properties.getProperty(INFORM_ABOUT_E_OR_U_IN_AUSSPRACHE);
            boolean informAboutEOrUInVocableFieldAusspracheAtStartup =
                    BooleanHandling.stringToBoolean(
                            informAboutEOrUInVocableFieldAusspracheAtStartupAsString);
            options.setInformAboutEOrUInVocableFieldAusspracheAtStartup(
                    informAboutEOrUInVocableFieldAusspracheAtStartup);
        }
        if (properties.containsKey(NUMBER_OF_REMEMBERED_OWN_LISTS)) {
            String numberOfRememberedOwnListsAsString =
                    properties.getProperty(NUMBER_OF_REMEMBERED_OWN_LISTS);
            int numberOfRememberedOwnLists =
                    NumberString.parseIntIgnore(numberOfRememberedOwnListsAsString, -1);
            options.setNumberOfRememberedOwnLists(numberOfRememberedOwnLists);
        }
        if (properties.containsKey(HISTORICAL_LEFT_OWN_LISTS_IN_LIST_COMMANDER)) {
            String historicalLeftOwnListsInListCommanderString =
                    properties.getProperty(HISTORICAL_LEFT_OWN_LISTS_IN_LIST_COMMANDER);
            options.setHistoricalLeftOwnListsInListCommander(
                    HistoricalOwnListPersistanceDataList.createFromPersistanceString(
                            historicalLeftOwnListsInListCommanderString, 1000));
        }
        if (properties.containsKey(HISTORICAL_RIGHT_OWN_LISTS_IN_LIST_COMMANDER)) {
            String historicalRightOwnListsInListCommanderString =
                    properties.getProperty(HISTORICAL_RIGHT_OWN_LISTS_IN_LIST_COMMANDER);
            options.setHistoricalRightOwnListsInListCommander(
                    HistoricalOwnListPersistanceDataList.createFromPersistanceString(
                            historicalRightOwnListsInListCommanderString, 1000));
        }
        if (properties.containsKey(HISTORICAL_OWN_LISTS)) {
            String historicalOwnListsString = properties.getProperty(HISTORICAL_OWN_LISTS);
            options.setHistoricalOwnLists(
                    HistoricalOwnListPersistanceDataList.createFromPersistanceString(
                            historicalOwnListsString, 1000));
        }
        if (properties.containsKey(SHOW_ALL_TRANSLATIONS)) {
            String showAllTranslationsAsString = properties.getProperty(SHOW_ALL_TRANSLATIONS);
            boolean showAllTranslations =
                    BooleanHandling.stringToBoolean(showAllTranslationsAsString);
            options.setShowAllTranslations(showAllTranslations);
        }
        if (properties.containsKey(NUMBER_OF_SHOWN_TRANSLATIONS)) {
            String numberOfShownTranslationsAsString =
                    properties.getProperty(NUMBER_OF_SHOWN_TRANSLATIONS);
            int numberOfShownTranslations =
                    NumberString.parseIntIgnore(numberOfShownTranslationsAsString, -1);
            options.setNumberOfShownTranslations(numberOfShownTranslations);
        }
        if (properties.containsKey(SHOW_NUMBER_OF_LISTS)) {
            String showNumberOfListsInListerAsString = properties.getProperty(SHOW_NUMBER_OF_LISTS);
            boolean showNumberOfListsInLister =
                    BooleanHandling.stringToBoolean(showNumberOfListsInListerAsString);
            options.setShowNumberOfListsInLister(showNumberOfListsInLister);
        }
        if (properties.containsKey(SHOW_BUTTONS_IN_EINZELDARSTELLUNG)) {
            String showDetailBottomButonsInEinzeldarstellungTooAsString =
                    properties.getProperty(SHOW_BUTTONS_IN_EINZELDARSTELLUNG);
            boolean showDetailBottomButonsInEinzeldarstellungToo =
                    BooleanHandling.stringToBoolean(showDetailBottomButonsInEinzeldarstellungTooAsString);
            options.setShowDetailBottomButonsInEinzeldarstellungToo(
                    showDetailBottomButonsInEinzeldarstellungToo);
        }
        if (properties.containsKey(SHOW_BUTTONS_IN_TRANSLATION_EVALUATION)) {
            String showDetailBottomButonsInTranslationEvaluationTooAsString =
                    properties.getProperty(SHOW_BUTTONS_IN_TRANSLATION_EVALUATION);
            boolean showDetailBottomButonsInTranslationEvaluationToo =
                    BooleanHandling.stringToBoolean(showDetailBottomButonsInTranslationEvaluationTooAsString);
            options.setShowDetailBottomButonsInTranslationEvaluationToo(
                    showDetailBottomButonsInTranslationEvaluationToo);
        }
        if (properties.containsKey(PLAY_MP3_WHEN_TESTING)) {
            String playMP3WhenTestingVocablesAsString =
                    properties.getProperty(PLAY_MP3_WHEN_TESTING);
            boolean playMP3WhenTestingVocables =
                    BooleanHandling.stringToBoolean(playMP3WhenTestingVocablesAsString);
            options.setPlayMP3WhenTestingVocables(playMP3WhenTestingVocables);
        }
        if (properties.containsKey(COLOR_USER_INPUT_WHEN_TESTING_KANJI)) {
            String colorUserInputAtKanjiTestEvaluationDialogAsString =
                    properties.getProperty(COLOR_USER_INPUT_WHEN_TESTING_KANJI);
            boolean colorUserInputAtKanjiTestEvaluationDialog =
                    BooleanHandling.stringToBoolean(
                            colorUserInputAtKanjiTestEvaluationDialogAsString);
            options.setColorUserInputAtKanjiTestEvaluationDialog(
                    colorUserInputAtKanjiTestEvaluationDialog);
        }
        if (properties.containsKey(STOP_AFTER_STARTUP)) {
            String stopAfterStartupAsString = properties.getProperty(STOP_AFTER_STARTUP);
            boolean stopAfterStartup = BooleanHandling.stringToBoolean(stopAfterStartupAsString);
            options.setStopAfterStartup(stopAfterStartup);
        }
        if (properties.containsKey(WAIT_FOR_X_AFTER_STARTUP)) {
            String waitForXAfterStartupString = properties.getProperty(WAIT_FOR_X_AFTER_STARTUP);
            boolean waitForXAfterStartup = BooleanHandling.stringToBoolean(
                    waitForXAfterStartupString);
            options.setWaitForXAfterStartup(waitForXAfterStartup);
        }
        if (properties.containsKey(GT_SHOWN_KANJI)) {
            String grammarTableShowKanjiString = properties.getProperty(GT_SHOWN_KANJI);
            boolean grammarTableShowKanji = BooleanHandling.stringToBoolean(
                    grammarTableShowKanjiString);
            options.setGrammarTableShowKanji(grammarTableShowKanji);
        }
        if (properties.containsKey(GT_SHOW_KANA)) {
            String grammarTableShowKanaString = properties.getProperty(GT_SHOW_KANA);
            boolean grammarTableShowKana = BooleanHandling.stringToBoolean(
                    grammarTableShowKanaString);
            options.setGrammarTableShowKana(grammarTableShowKana);
        }
        if (properties.containsKey(GT_SHOW_ROMAJI)) {
            String grammarTableShowRomajiString = properties.getProperty(GT_SHOW_ROMAJI);
            boolean grammarTableShowRomaji = BooleanHandling.stringToBoolean(
                    grammarTableShowRomajiString);
            options.setGrammarTableShowRomaji(grammarTableShowRomaji);
        }
        if (properties.containsKey(GT_SEARCH_BASE_FORMS)) {
            String grammarTableSearchForBaseFormsString =
                    properties.getProperty(GT_SEARCH_BASE_FORMS);
            boolean grammarTableSearchForBaseForms = BooleanHandling.stringToBoolean(
                    grammarTableSearchForBaseFormsString);
            options.setGrammarTableSearchForBaseForms(grammarTableSearchForBaseForms);
        }
        if (properties.containsKey(GT_LEAVE_BLANK)) {
            String grammarTableLeaveNotFoundFormsBlankString =
                    properties.getProperty(GT_LEAVE_BLANK);
            boolean grammarTableLeaveNotFoundFormsBlank = BooleanHandling.stringToBoolean(
                    grammarTableLeaveNotFoundFormsBlankString);
            options.setGrammarTableLeaveNotFoundFormsBlank(grammarTableLeaveNotFoundFormsBlank);
        }
        if (properties.containsKey(GT_VERB_HEADERS)) {
            String grammarTableVerbHeadersString = properties.getProperty(GT_VERB_HEADERS);
            Map<String, String> map = GrammarTableHeaderPersistenceHelper.
                    createGrammarTableHeaderMapFormPersistenceString(grammarTableVerbHeadersString);
            Map<String, String> grammarTableVerbHeadersMap =
                    options.getGrammarTableVerbHeadersMap();
            grammarTableVerbHeadersMap.clear();
            grammarTableVerbHeadersMap.putAll(map);
        }
        if (properties.containsKey(GT_I_ADJECTIVE_HEADERS)) {
            String grammarTableIAdjectiveHeadersString = properties.getProperty(GT_I_ADJECTIVE_HEADERS);
            Map<String, String> map = GrammarTableHeaderPersistenceHelper.
                    createGrammarTableHeaderMapFormPersistenceString(
                            grammarTableIAdjectiveHeadersString);
            Map<String, String> grammarTableIAdjectiveHeadersMap =
                    options.getGrammarTableIAdjectiveHeadersMap();
            grammarTableIAdjectiveHeadersMap.clear();
            grammarTableIAdjectiveHeadersMap.putAll(map);
        }
        if (properties.containsKey(GT_NA_ADJECTIVE_HEADERS)) {
            String grammarTableNaAdjectiveHeadersString =
                    properties.getProperty(GT_NA_ADJECTIVE_HEADERS);
            Map<String, String> map = GrammarTableHeaderPersistenceHelper.
                    createGrammarTableHeaderMapFormPersistenceString(
                            grammarTableNaAdjectiveHeadersString);
            Map<String, String> grammarTableNaAdjectiveHeadersMap =
                    options.getGrammarTableNaAdjectiveHeadersMap();
            grammarTableNaAdjectiveHeadersMap.clear();
            grammarTableNaAdjectiveHeadersMap.putAll(map);
        }
        if (properties.containsKey(GT_USE_USER_HEADERS)) {
            String grammarTableUseUserHeadersString = properties.getProperty(GT_USE_USER_HEADERS);
            boolean grammarTableUseUserHeaders = BooleanHandling.stringToBoolean(
                    grammarTableUseUserHeadersString);
            options.setGrammarTableUseUserHeaders(grammarTableUseUserHeaders);
        }
        if (properties.containsKey(GT_USER_HEADER_SAVE_LOAD_DIR)) {
            String grammarTableLastUsedOwnHeadersDirectory =
                    properties.getProperty(GT_USER_HEADER_SAVE_LOAD_DIR);
            options.setGrammarTableLastUsedOwnHeadersDirectory(
                    grammarTableLastUsedOwnHeadersDirectory);
        }
        if (properties.containsKey(CLOSE_SINGLE_VIEW_WHEH_SHOWING_DETAILS)) {
            String closeSingleViewWhenOpeningDetailsAfterSimpleSearchString =
                    properties.getProperty(CLOSE_SINGLE_VIEW_WHEH_SHOWING_DETAILS);
            boolean closeSingleViewWhenOpeningDetailsAfterSimpleSearch =
                    BooleanHandling.stringToBoolean(
                            closeSingleViewWhenOpeningDetailsAfterSimpleSearchString);
            options.setCloseSingleViewWhenOpeningDetailsAfterSimpleSearch(
                    closeSingleViewWhenOpeningDetailsAfterSimpleSearch);
        }
        if (properties.containsKey(SHOW_WARNUNGS_AFTER_START)) {
            String showWarningsAfterStartSearchString =
                    properties.getProperty(SHOW_WARNUNGS_AFTER_START);
            boolean showWarningsAfterStart =
                    BooleanHandling.stringToBoolean(showWarningsAfterStartSearchString);
            options.setShowWarningsAfterStart(showWarningsAfterStart);
        }
        if (properties.containsKey(SHOW_KANJI_TEST_AT_PARAMETER_POSITION)) {
            String showKanjiTestAtParameterDialogPositionString =
                    properties.getProperty(SHOW_KANJI_TEST_AT_PARAMETER_POSITION);
            boolean showKanjiTestAtParameterDialogPosition =
                    BooleanHandling.stringToBoolean(showKanjiTestAtParameterDialogPositionString);
            options.setShowKanjiTestAtParameterDialogPosition(
                    showKanjiTestAtParameterDialogPosition);
        }

        if (properties.containsKey(SHOW_KANJI_NUMBER_IN_NOTENOOK)) {
            String showKanjiNumberInMyHandwrittenKanjiNotebooksString =
                    properties.getProperty(SHOW_KANJI_NUMBER_IN_NOTENOOK);
            boolean showKanjiNumberInMyHandwrittenKanjiNotebooks =
                    BooleanHandling.stringToBoolean(
                            showKanjiNumberInMyHandwrittenKanjiNotebooksString);
            options.setShowKanjiNumberInMyHandwrittenKanjiNotebooks(
                    showKanjiNumberInMyHandwrittenKanjiNotebooks);
        }
        if (properties.containsKey(NUMBER_OF_CHARS_IN_BAR_TITLE)) {
            String numberOfCharsInVocableBarTitleShownAsString =
                    properties.getProperty(NUMBER_OF_CHARS_IN_BAR_TITLE);
            int numberOfCharsInVocableBarTitleShown = NumberString.parseIntIgnore(
                    numberOfCharsInVocableBarTitleShownAsString, -1);
            options.setNumberOfCharsInVocableBarTitleShown(numberOfCharsInVocableBarTitleShown);
        }
    }

    /**
     * Da man beim Laden nicht weiß, ob die Anzahl oder die historischen Listen der eigenen Listen
     * im Listen-Commander zuerst geladen werden, werden die Listen mit 1000 Plätzen
     * vorinitialisiert und am Ende auf den richtigen Wert gesetzt.
     */
    private void adjustHistoricalOwnListsInListCommander() {
        int numberOfRememberedOwnLists = options.getNumberOfRememberedOwnLists();
        HistoricalOwnListPersistanceDataList leftDataList =
                options.getHistoricalLeftOwnListsInListCommander();
        HistoricalOwnListPersistanceDataList rightDataList =
                options.getHistoricalRightOwnListsInListCommander();
        HistoricalOwnListPersistanceDataList dataList =
                options.getHistoricalOwnLists();
        leftDataList.setNumberOfRememberedOwnLists(numberOfRememberedOwnLists);
        rightDataList.setNumberOfRememberedOwnLists(numberOfRememberedOwnLists);
        dataList.setNumberOfRememberedOwnLists(numberOfRememberedOwnLists);
    }

    /** Speichert die Optionen. */
    public void save() {
        Properties properties = new Properties();
        saveOptions(properties);
        helper.writeProperties(properties);
    }

    /**
     * Speichert die Optionen. Hier werden die eigentlichen Daten in die Properties geschrieben.
     */
    private void saveOptions(Properties properties) {
        properties.setProperty(STRING_SELECTION_FONT_SIZE,
                Integer.toString(options.getStringSelectionFontSize()));
        properties.setProperty(STRING_SELECTION_LABEL_FONT_SIZE,
                Integer.toString(options.getStringSelectionLabelFontSize()));
        properties.setProperty(LAST_USED_VERSION, vocabularyTrainerVersion.toSerializedString());
        properties.setProperty(VOCABULARIES_PATH, options.getVocabulariesPath());
        properties.setProperty(SWITCH_TO_NEXT_VOCABLE,
                BooleanHandling.booleanToString(
                        options.isSwitchToNextVocableAfterEntringTranslation()));
        properties.setProperty(SHOW_ROMAJI,
                BooleanHandling.booleanToString(
                        options.isShowRomajiWhenTestingVocable()));
        properties.setProperty(SHOW_PRONUNCIATION,
                BooleanHandling.booleanToString(
                        options.isShowPronunciationWhenTestingVocable()));
        properties.setProperty(SHOW_COMMENT,
                BooleanHandling.booleanToString(
                        options.isShowCommentWhenTestingVocable()));
        properties.setProperty(SHOW_VOCABULARY_DESCRIPTION,
                BooleanHandling.booleanToString(
                        options.isShowVocabularyDescriptionWhenTestingVocable()));
        properties.setProperty(BACKGROUND_COLORING_TESTING,
                BooleanHandling.booleanToString(
                        options.isColorVocableDependingOnLastSuccessWhenTestingVocable()));
        properties.setProperty(BACKGROUND_COLORING_VIEWING,
                BooleanHandling.booleanToString(
                        options.isColorVocableDependingOnLastSuccessWhenViewingVocable()));
        properties.setProperty(VOCABULARY_SORT_ORDER,
                options.getVocabularySortOrder().name());
        properties.setProperty(FOREGROUND_COLORING_VOCABULARY,
                BooleanHandling.booleanToString(options.isColorVocabularyDependingOnLastSuccess()));
        properties.setProperty(VOCABLES_SHUFFLE_TYPE,
                options.getVocablesShuffleType().name());
        properties.setProperty(VOCABLES_SUBSET_SELECTION_TYPE,
                options.getVocablesSubsetSelectionType().name());
        properties.setProperty(REVERSE_SHUFFLED_VOCABLES,
                BooleanHandling.booleanToString(options.isReverseShuffledVocables()));
        properties.setProperty(USE_VOCABLES_SHUFFLE_TYPE_FOR_LISTS,
                BooleanHandling.booleanToString(
                        options.isUseVocablesShuffleTypeForShowingListsAndSheetsToo()));
        properties.setProperty(HIDE_START_OF_VOCABULARY,
                BooleanHandling.booleanToString(options.isHideStartOfVocabularyDescription()));
        properties.setProperty(USE_TEST_ORDER_FOR_VIEWING,
                BooleanHandling.booleanToString(options.isUseTestOrderForViewingVocables()));
        properties.setProperty(LAST_SHOWN_MAIN_TAB_INDEX,
                Integer.toString(options.getLastShownMainTabIndex()));
        properties.setProperty(LAST_SHOWN_VOCABULARY_CATEGORY_TAB_INDEX,
                Integer.toString(options.getLastShownVocabularyCategoryTabIndex()));
        properties.setProperty(LAST_SHOWN_OWN_LIST_CATEGORY_TAB_INDEX,
                Integer.toString(options.getLastShownOwnListCategoryTabIndex()));
        properties.setProperty(LAST_SHOWN_VOCABULARY_SUB_CATEGORY_TAB_INDEX,
                Integer.toString(options.getLastShownVocabularySubCategoryTabIndex()));
        properties.setProperty(LAST_SHOWN_OWN_LIST_SUB_CATEGORY_TAB_INDEX,
                Integer.toString(options.getLastShownOwnListSubCategoryTabIndex()));
        properties.setProperty(SHOW_SUCCESS_PERCENT_IN_VOCABULARY_BAR,
                BooleanHandling.booleanToString(options.isShowSuccessPercentInVocabularyBar()));
        properties.setProperty(NUMBER_OF_VOCABULARY_BAR_COLUMNS,
                Integer.toString(options.getNumberOfVocabularyBarColumns()));
        properties.setProperty(LAST_USED_OWN_LIST_DIRECTORY,
                options.getLastUsedOwnListImportExportDirectory());
        properties.setProperty(NUMBER_OF_DATASETS_PER_PAGE_OLE,
                Integer.toString(options.getNumberOfDatasetsPerPageOfOwnListEditor()));
        properties.setProperty(NUMBER_OF_COLUMNS_OLE,
                Integer.toString(options.getNumberOfColumnsOfOwnListEditor()));
        properties.setProperty(LOCATION_OF_OLE,
                SwingSessionManagerHelper.createPointStorageString(
                        options.getLocationOfOwnListEditor()));
        properties.setProperty(SIZE_OF_OLE,
                SwingSessionManagerHelper.createDimensionStorageString(
                        options.getSizeOfOwnListEditor()));
        properties.setProperty(HIDE_VOCABLES_FROM_LIST_OLE,
                BooleanHandling.booleanToString(
                        options.isHideVocablesFromListInSearchAreaInOwnListEditor()));
        properties.setProperty(ALLOW_MISSING_MP3,
                BooleanHandling.booleanToString(options.isAllowMissingMp3()));
        properties.setProperty(REPORT_MISSING_MP3,
                BooleanHandling.booleanToString(options.isReportMissingMp3()));
        properties.setProperty(NUMBER_OF_COLUMNS_OF_BLATT,
                Integer.toString(options.getNumberOfColumnsOfBlattDarstellung()));
        properties.setProperty(INFORM_ABOUT_DOUBLE_MP3,
                BooleanHandling.booleanToString(options.isInformAboutDoubleMp3AtStartup()));
        properties.setProperty(ONLY_USE_FIRST_GERMAN_TERM_TESTING,
                BooleanHandling.booleanToString(
                        options.isUseOnlyFirstGermanTermWhenTestingVocableGermanToJapanese()));
        properties.setProperty(TRANSLATION_DIRECTION, options.getTranslationDirection().name());
        properties.setProperty(SHOW_KANJI,
                BooleanHandling.booleanToString(options.isShowKanjiWhenTestingVocable()));
        properties.setProperty(HIDE_KANJI_IF_HIRAGANA,
                BooleanHandling.booleanToString(
                        options.isHideKanjiWhenTestingVocableAndKanaContainsOnlyHiragana()));
        properties.setProperty(HIDE_ROMAJI_IF_HIRAGANA,
                BooleanHandling.booleanToString(
                        options.isHideRomajiWhenTestingVocableAndKanaContainsOnlyHiragana()));
        properties.setProperty(HIDE_PRONUNCIATION_IF_HIRAGANA,
                BooleanHandling.booleanToString(
                        options.isHidePronunciationWhenTestingVocableAndKanaContainsOnlyHiragana()));
        properties.setProperty(SHOW_DETAILS_BUTTON_EINZELDARSTELLUNG,
                BooleanHandling.booleanToString(
                        options.isShowDetailsButtonInEinzeldarstellung()));
        properties.setProperty(SHOW_DETAILS_BUTTON_VOKABELBLATTDARSTELLUNG,
                BooleanHandling.booleanToString(
                        options.isShowDetailsButtonInVokabelblattDarstellung()));
        properties.setProperty(HIDE_LESS_START_OF_VOCABULARY_DESCRIPTION_VHS,
                BooleanHandling.booleanToString(
                        options.isHideLessStartOfVocabularyDescriptionForVhs()));
        properties.setProperty(INFORM_ABOUT_EQUAL_KANA_AND_KANJI,
                BooleanHandling.booleanToString(options.isInformAboutEqualKanaAndKanjiAtStartup()));
        properties.setProperty(SHOW_KANA,
                BooleanHandling.booleanToString(options.isShowKanaWhenTestingVocable()));
        properties.setProperty(CHECK_NEW_LISTS,
                BooleanHandling.booleanToString(options.isCheckNewOwnListsVersionAtStart()));
        properties.setProperty(KANA_IN_KANJI_SIZE_WHEN_TESTING,
                BooleanHandling.booleanToString(options.isShowKanaInKanjiSizeWhenTestingVocable()));
        properties.setProperty(EMTPY_TRANSLATION_ACCEPTANCE,
                options.getEmptyTranslationsAcceptance().toPersistence());
        properties.setProperty(NUMBER_OF_DAYS_FOR_A_WHILE,
                Integer.toString(options.getNumberOfDaysForAWhile()));
        properties.setProperty(NUMBER_OF_DAYS_A_VOCABLE_IS_NEW,
                Integer.toString(options.getNumberOfDaysAVocableIsNew()));
        properties.setProperty(COLORS,
                SwingSessionManagerHelper.createColorsStorageString(options.getColors()));
        properties.setProperty(DELTA_FOREGROUND,
                Integer.toString(options.getDeltaForForegroundColor()));
        properties.setProperty(DELTA_BACKGROUND,
                Integer.toString(options.getDeltaForBackgroundColor()));
        properties.setProperty(COLOR_MODIFICATION_TYPE,
                options.getColorModificationType().name());
        properties.setProperty(USE_SPLIT_PANE_IN_OLE,
                BooleanHandling.booleanToString(options.isUseSplitPaneInOwnListEditor()));
        properties.setProperty(VOCABLE_COLOR_STANDARD,
                SwingSessionManagerHelper.createColorStorageString(
                        options.getVocableColorStandard()));
        properties.setProperty(VOCABLE_COLOR_KANJI,
                SwingSessionManagerHelper.createColorStorageString(
                        options.getVocableColorKanji()));
        properties.setProperty(VOCABLE_COLOR_KANA,
                SwingSessionManagerHelper.createColorStorageString(
                        options.getVocableColorKana()));
        properties.setProperty(VOCABLE_COLOR_ROMAJI,
                SwingSessionManagerHelper.createColorStorageString(
                        options.getVocableColorRomaji()));
        properties.setProperty(VOCABLE_COLOR_PRONUNCIATION,
                SwingSessionManagerHelper.createColorStorageString(
                        options.getVocableColorPronunciation()));
        properties.setProperty(VOCABLE_COLOR_TRANSLATION,
                SwingSessionManagerHelper.createColorStorageString(
                        options.getVocableColorTranslation()));
        properties.setProperty(VOCABLE_COLOR_COMMENT,
                SwingSessionManagerHelper.createColorStorageString(
                        options.getVocableColorComment()));
        properties.setProperty(VOCABLE_COLOR_SEARCH_WORDS,
                SwingSessionManagerHelper.createColorStorageString(
                        options.getVocableColorSearchWords()));
        properties.setProperty(VOCABLE_COLOR_PART_OF_SPEACH,
                SwingSessionManagerHelper.createColorStorageString(
                        options.getVocableColorPartOfSpeach()));
        properties.setProperty(SHOW_STARTUP_TIMESTAMP,
                BooleanHandling.booleanToString(options.isShowTimestampsInStartup()));
        properties.setProperty(CREATE_MAP_D_TO_J_AT_STARTUP,
                BooleanHandling.booleanToString(
                        options.isCreateGermanJapaneseTranslationAtStartup()));
        properties.setProperty(LAST_USED_7ZIP_DIR,
                options.getLastUsed7ZipDirectory());
        properties.setProperty(LAST_SELECTED_KANJI_SET_TEXT,
                options.getLastSelectedKanjiSetTextInCombobox());
        properties.setProperty(INFORM_ABOUT_E_OR_U_IN_AUSSPRACHE,
                BooleanHandling.booleanToString(
                        options.isInformAboutEOrUInVocableFieldAusspracheAtStartup()));
        properties.setProperty(NUMBER_OF_REMEMBERED_OWN_LISTS,
                Integer.toString(options.getNumberOfRememberedOwnLists()));
        properties.setProperty(HISTORICAL_LEFT_OWN_LISTS_IN_LIST_COMMANDER,
                options.getHistoricalLeftOwnListsInListCommander().toPersistanceString());
        properties.setProperty(HISTORICAL_RIGHT_OWN_LISTS_IN_LIST_COMMANDER,
                options.getHistoricalRightOwnListsInListCommander().toPersistanceString());
        properties.setProperty(HISTORICAL_OWN_LISTS,
                options.getHistoricalOwnLists().toPersistanceString());
        properties.setProperty(SHOW_ALL_TRANSLATIONS,
                BooleanHandling.booleanToString(options.isShowAllTranslations()));
        properties.setProperty(NUMBER_OF_SHOWN_TRANSLATIONS,
                Integer.toString(options.getNumberOfShownTranslations()));
        properties.setProperty(SHOW_NUMBER_OF_LISTS,
                BooleanHandling.booleanToString(
                        options.isShowNumberOfListsInLister()));
        properties.setProperty(SHOW_BUTTONS_IN_EINZELDARSTELLUNG,
                BooleanHandling.booleanToString(
                        options.isShowDetailBottomButonsInEinzeldarstellungToo()));
        properties.setProperty(SHOW_BUTTONS_IN_TRANSLATION_EVALUATION,
                BooleanHandling.booleanToString(
                        options.isShowDetailBottomButonsInTranslationEvaluationToo()));
        properties.setProperty(PLAY_MP3_WHEN_TESTING,
                BooleanHandling.booleanToString(options.isPlayMP3WhenTestingVocables()));
        properties.setProperty(COLOR_USER_INPUT_WHEN_TESTING_KANJI,
                BooleanHandling.booleanToString(
                        options.isColorUserInputAtKanjiTestEvaluationDialog()));
        properties.setProperty(STOP_AFTER_STARTUP,
                BooleanHandling.booleanToString(options.isStopAfterStartup()));
        properties.setProperty(WAIT_FOR_X_AFTER_STARTUP,
                BooleanHandling.booleanToString(options.isWaitForXAfterStartup()));
        properties.setProperty(GT_SHOWN_KANJI,
                BooleanHandling.booleanToString(options.isGrammarTableShowKanji()));
        properties.setProperty(GT_SHOW_KANA,
                BooleanHandling.booleanToString(options.isGrammarTableShowKana()));
        properties.setProperty(GT_SHOW_ROMAJI,
                BooleanHandling.booleanToString(options.isGrammarTableShowRomaji()));
        properties.setProperty(GT_SEARCH_BASE_FORMS,
                BooleanHandling.booleanToString(options.isGrammarTableSearchForBaseForms()));
        properties.setProperty(GT_LEAVE_BLANK,
                BooleanHandling.booleanToString(options.isGrammarTableLeaveNotFoundFormsBlank()));
        properties.setProperty(GT_VERB_HEADERS, GrammarTableHeaderPersistenceHelper.
                createGrammarTableHeaderPesistenceString(options.getGrammarTableVerbHeadersMap()));
        properties.setProperty(GT_I_ADJECTIVE_HEADERS, GrammarTableHeaderPersistenceHelper.
                createGrammarTableHeaderPesistenceString(
                        options.getGrammarTableIAdjectiveHeadersMap()));
        properties.setProperty(GT_NA_ADJECTIVE_HEADERS, GrammarTableHeaderPersistenceHelper.
                createGrammarTableHeaderPesistenceString(
                        options.getGrammarTableNaAdjectiveHeadersMap()));
        properties.setProperty(GT_USE_USER_HEADERS,
                BooleanHandling.booleanToString(options.isGrammarTableUseUserHeaders()));
        properties.setProperty(GT_USER_HEADER_SAVE_LOAD_DIR,
                options.getGrammarTableLastUsedOwnHeadersDirectory());
        properties.setProperty(CLOSE_SINGLE_VIEW_WHEH_SHOWING_DETAILS,
                BooleanHandling.booleanToString(
                        options.isCloseSingleViewWhenOpeningDetailsAfterSimpleSearch()));
        properties.setProperty(SHOW_WARNUNGS_AFTER_START,
                BooleanHandling.booleanToString(options.isShowWarningsAfterStart()));
        properties.setProperty(SHOW_KANJI_TEST_AT_PARAMETER_POSITION,
                BooleanHandling.booleanToString(options.isShowKanjiTestAtParameterDialogPosition()));
        properties.setProperty(SHOW_KANJI_NUMBER_IN_NOTENOOK,
                BooleanHandling.booleanToString(
                        options.isShowKanjiNumberInMyHandwrittenKanjiNotebooks()));
        properties.setProperty(NUMBER_OF_CHARS_IN_BAR_TITLE,
                Integer.toString(options.getNumberOfCharsInVocableBarTitleShown()));
    }

}
