package de.duehl.vocabulary.japanese.data;

import java.util.List;

import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.logic.internal.InternalDataRequester;
import de.duehl.vocabulary.japanese.logic.ownlists.OwnLists;
import de.duehl.vocabulary.japanese.logic.symbol.kana.internal.InternalKanaDataRequester;
import de.duehl.vocabulary.japanese.logic.symbol.kanji.internal.InternalKanjiDataRequester;
import de.duehl.vocabulary.japanese.logic.translation.GermanToJapaneseTranslation;
import de.duehl.vocabulary.japanese.logic.translation.JapaneseToGermanTranslation;
import de.duehl.vocabulary.japanese.logic.wrongtested.WrongTestedVocables;

/**
 * Diese Klasse steht für alle feststehenden und veränderlichen Datenstrukturen des Vokabeltrainers.
 *
 * @version 1.01     2025-11-20
 * @author Christian Dühl
 */

public class FumikoDataStructures {

    /** Die Programmoptionen. */
    private final Options options;

    /** Die Liste mit den bekannten Vokabularien. */
    private final List<Vocabulary> vocabularies;

    /** Das Objekt das zu Vokabeln die internen Daten abfragt. */
    private final InternalDataRequester internalDataRequester;

    /**
     * Das Objekt das sich um die Übersetzung von Japanisch in Deutsch kümmert (für Abfragen die
     * nur Kana, aber nicht Kanji anzeigen).
     */
    private final JapaneseToGermanTranslation japaneseToGermanTranslation;

    /** Das Objekt das sich um die Übersetzung von Deutsch in Japanisch kümmert. */
    private GermanToJapaneseTranslation germanToJapaneseTranslation;

    /** Die Verwaltung der eigenen Vokabellisten. */
    private final OwnLists ownLists;

    /**
     * Die Verwaltung der beiden automatisch gepflegten Listen mit falsch abgefragten Vokabeln aus
     * Gruppen und anderen Vokabularien.
     */
    private final WrongTestedVocables wrongTestedVocables;

    /** Das Objekt, das zu einem Kanji die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalKanjiDataRequester internalKanjiDataRequester;

    /** Die Liste der benutzerdefinierten Kanji-Mengen. */
    private final List<KanjiSet> kanjiSets;

    /** Das Objekt das zu einem Kana die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalKanaDataRequester internalKanaDataRequester;

    /**
     * Konstruktor.
     *
     * @param options
     *            Die Programmoptionen.
     * @param vocabularies
     *            Die Liste mit den bekannten Vokabularien.
     * @param internalDataRequester
     *            Das Objekt das zu Vokabeln die internen Daten abfragt.
     * @param japaneseToGermanTranslation
     *            Das Objekt das sich um die Übersetzung von Japanisch in Deutsch kümmert (für
     *            Abfragen die nur Kana, aber nicht Kanji anzeigen).
     * @param germanToJapaneseTranslation
     *            Das Objekt das sich um die Übersetzung von Deutsch in Japanisch kümmert.
     * @param ownLists
     *            Die Verwaltung der eigenen Vokabellisten.
     * @param wrongTestedVocables
     *            Die Verwaltung der beiden automatisch gepflegten Listen mit falsch abgefragten
     *            Vokabeln aus Gruppen und anderen Vokabularien.
     * @param internalKanjiDataRequester
     *            Das Objekt, das zu einem Kanji die internen, benutzerabhängigen Daten abrufen
     *            kann.
     * @param kanjiSets
     *            Die Liste der benutzerdefinierten Kanji-Mengen.
     * @param internalKanaDataRequester
     *            Das Objekt das zu einem Kana die internen, benutzerabhängigen Daten abrufen kann.
     */
    public FumikoDataStructures(Options options, List<Vocabulary> vocabularies,
            InternalDataRequester internalDataRequester,
            JapaneseToGermanTranslation japaneseToGermanTranslation,
            GermanToJapaneseTranslation germanToJapaneseTranslation, OwnLists ownLists,
            WrongTestedVocables wrongTestedVocables,
            InternalKanjiDataRequester internalKanjiDataRequester, List<KanjiSet> kanjiSets,
            InternalKanaDataRequester internalKanaDataRequester) {
        this.options = options;
        this.vocabularies = vocabularies;
        this.internalDataRequester = internalDataRequester;
        this.japaneseToGermanTranslation = japaneseToGermanTranslation;
        this.germanToJapaneseTranslation = germanToJapaneseTranslation;
        this.ownLists = ownLists;
        this.wrongTestedVocables = wrongTestedVocables;
        this.internalKanjiDataRequester = internalKanjiDataRequester;
        this.kanjiSets = kanjiSets;
        this.internalKanaDataRequester = internalKanaDataRequester;
    }

    /** Getter für die Programmoptionen. */
    public Options getOptions() {
        return options;
    }

    /** Getter für die Liste mit den bekannten Vokabularien. */
    public List<Vocabulary> getVocabularies() {
        return vocabularies;
    }

    /** Getter für das Objekt, das zu Vokabeln die internen Daten abfragt. */
    public InternalDataRequester getInternalDataRequester() {
        return internalDataRequester;
    }

    /**
     * Getter für das Objekt das sich um die Übersetzung von Japanisch in Deutsch kümmert (für
     * Abfragen die nur Kana, aber nicht Kanji anzeigen).
     */
    public JapaneseToGermanTranslation getJapaneseToGermanTranslation() {
        return japaneseToGermanTranslation;
    }

    /** Getter für das Objekt das sich um die Übersetzung von Deutsch in Japanisch kümmert. */
    public GermanToJapaneseTranslation getGermanToJapaneseTranslation() {
        return germanToJapaneseTranslation;
    }

    /**
     * Setter für das Objekt das sich um die Übersetzung von Deutsch in Japanisch kümmert.
     *
     * Für den Fall, dass diese nicht beim Startup erzeugt werden.
     */
    public void setGermanToJapaneseTranslation(
            GermanToJapaneseTranslation germanToJapaneseTranslation) {
        this.germanToJapaneseTranslation = germanToJapaneseTranslation;
    }

    /** Getter für die Verwaltung der eigenen Vokabellisten. */
    public OwnLists getOwnLists() {
        return ownLists;
    }

    /**
     * Getter für die Verwaltung der beiden automatisch gepflegten Listen mit falsch abgefragten
     * Vokabeln aus Gruppen und anderen Vokabularien.
     */
    public WrongTestedVocables getWrongTestedVocables() {
        return wrongTestedVocables;
    }

    /**
     * Getter für das Objekt, das zu einem Kanji die internen, benutzerabhängigen Daten abrufen
     * kann.
     */
    public InternalKanjiDataRequester getInternalKanjiDataRequester() {
        return internalKanjiDataRequester;
    }

    /** Getter für die Liste der benutzerdefinierten Kanji-Mengen. */
    public List<KanjiSet> getKanjiSets() {
        return kanjiSets;
    }

    /**
     * Getter für das Objekt das zu einem Kana die internen, benutzerabhängigen Daten abrufen kann.
     */
    public InternalKanaDataRequester getInternalKanaDataRequester() {
        return internalKanaDataRequester;
    }

}
