package de.duehl.vocabulary.japanese.io;

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.io.Charset;
import de.duehl.basics.io.FileHelper;
import de.duehl.vocabulary.japanese.common.data.InternalAdditionalVocableData;
import de.duehl.vocabulary.japanese.data.OwnList;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.logic.internal.InternalDataRequester;

/**
 * Diese Klasse schreibt eine Datei mit einer eigenen Liste ein.
 *
 * @version 1.01     2024-08-06
 * @author Christian Dühl
 */

public class OwnListWriter {

    /** Die zu speichernde eigene Liste. */
    private final OwnList ownList;

    /** Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen kann. */
    private final InternalDataRequester requester;

    /** Der Dateiname der eigenen Liste (wird im Normalfall aus dem Namen gebildet). */
    private String filename;

    /**
     * Gibt an, ob ein spezieller Dateiname (außerhalb des Verzeichnisses mit den eigenen Listen)
     * für die automatisch gepflegten Listen der falsch abgefragten Vokabeln verwendet werden soll.
     */
    private boolean specialFilenameForWrongTestedVocables;

    /**
     * Konstruktor.
     *
     * @param ownList
     *            Die zu speichernde eigene Liste.
     * @param requester
     *            Das Objekt das zu einer Vokabel die internen, benutzerabhängigen Daten abrufen
     *            kann.
     */
    public OwnListWriter(OwnList ownList, InternalDataRequester requester) {
        this.ownList = ownList;
        this.requester = requester;

        specialFilenameForWrongTestedVocables = false;
    }

    /**
     * Legt fest, dass ein spezieller Dateiname (außerhalb des Verzeichnisses mit den eigenen
     * Listen) für die automatisch gepflegten Listen der falsch abgefragten Vokabeln verwendet
     * werden soll.
     *
     * @param filename
     *            Der Dateiname der eigenen Liste.
     */
    public void useSpecialFilenameForWrongTestedVocables(String filename) {
        this.filename = filename;
        specialFilenameForWrongTestedVocables = true;
    }


    /** Speichert die eigene Liste. */
    public void write() {
        createFilename();
        writeOwnList();
    }

    private void createFilename() {
        if (!specialFilenameForWrongTestedVocables) {
            filename = ownList.determineFilename();
        }
    }

    private void writeOwnList() {
        List<String> lines = new ArrayList<>();

        lines.add(OwnListReader.OWN_LIST_CATEGORY_KEY + " = " + ownList.getCategory());
        lines.add(OwnListReader.OWN_LIST_SUB_CATEGORY_KEY + " = " + ownList.getSubCategory());

        for (Vocable vocable : ownList.getVocables()) {
            InternalAdditionalVocableData data = requester.getInternalDataForVocable(vocable);
            String key = data.getKey();
            lines.add(key);
        }

        FileHelper.writeLinesToFile(lines, filename, Charset.UTF_8);
    }

}
