package de.duehl.vocabulary.japanese.ui.components.text;

import de.duehl.basics.text.Text;
import de.duehl.vocabulary.japanese.data.Vocable;

/**
 * Diese Klasse erstellt den Text zu Anzeige für Kanji und Kana einer Vokabel.
 *
 * @version 1.01     2024-04-10
 * @author Christian Dühl
 */

public class KanjiAndKanaTextCreator {

    private static final int CHARACTERS_PER_LINE = 15;

    /** Die Vokabel, welche angezeigt wird. */
    private final Vocable vocable;

    /** Gibt an, ob die Kana dargestellt werden sollen. */
    private final boolean showKana;

    /** Gibt an, ob die Kanji dargestellt werden sollen. */
    private final boolean showKanji;

    /** Der Haupt-Text, enthält Kanji oder Kana, falls kein Kanji gesetzt ist. */
    private String firstTextPart;

    /**
     * Der Text, der in Klammern angezeigt werden soll, also normalerweise Kana. Wenn man den nicht
     * braucht, ist er leer. Man muss ihn noch mit Klammern umgeben, wenn er nicht leer ist.
     */
    private String textPartInBrace;

    /** Gibt an, ob die Klammer in einer eigenen Zeile dargestellt werden soll. */
    private boolean braceInOwnLine;

    /**
     * Konstruktor.
     *
     * @param vocable
     *            Die Vokabel, welche angezeigt wird.
     * @param showKana
     *            Gibt an, ob die Kana dargestellt werden sollen.
     * @param showKanji
     *            Gibt an, ob die Kanji dargestellt werden sollen.
     */
    public KanjiAndKanaTextCreator(Vocable vocable, boolean showKana, boolean showKanji) {
        this.vocable = vocable;
        this.showKana = showKana;
        this.showKanji = showKanji;
    }

    /** Erzeugt den Text für die Anzeige von Kanji und Kana. */
    public void create() {
        initializeTexts();
        breakLongTexts();
    }

    private void initializeTexts() {
        firstTextPart = vocable.getKanji();
        textPartInBrace = vocable.getKana();

        if (!showKana) {
            textPartInBrace = "";
        }

        if (firstTextPart.isBlank() || !showKanji) {
            firstTextPart = textPartInBrace;
            textPartInBrace = "";
        }
        else if (firstTextPart.equals(textPartInBrace)) {
            textPartInBrace = "";
        }

        braceInOwnLine = !textPartInBrace.isEmpty()
                && firstTextPart.length() + textPartInBrace.length() + 3 > CHARACTERS_PER_LINE;
                // 3: Klammern, Leerzeichen
    }

    private void breakLongTexts() {
        firstTextPart = breakLongText(firstTextPart);
        textPartInBrace = breakLongText(textPartInBrace);
    }

    private String breakLongText(String text) {
        if (text.length() > CHARACTERS_PER_LINE) {
            return Text.addLineBreaks(text, CHARACTERS_PER_LINE);
        }
        else {
            return text;
        }
    }

    /** Getter für den Haupt-Text, enthält Kanji oder Kana, falls kein Kanji gesetzt ist. */
    public String getFirstTextPart() {
        return firstTextPart;
    }

    /**
     * Getter für den Text, der in Klammern angezeigt werden soll, also normalerweise Kana. Wenn
     * man den nicht braucht, ist er leer. Man muss ihn noch mit Klammern umgeben, wenn er nicht
     * leer ist.
     */
    public String getTextPartInBrace() {
        return textPartInBrace;
    }

    /** Gibt an, ob die Klammer in einer eigenen Zeile dargestellt werden soll. */
    public boolean isBraceInOwnLine() {
        return braceInOwnLine;
    }

}
