package de.duehl.vocabulary.japanese.ui.dialog.ownlist;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;

import de.duehl.basics.text.html.HtmlTool;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.FumikoDataStructures;
import de.duehl.vocabulary.japanese.data.OwnList;
import de.duehl.vocabulary.japanese.ui.data.FumikoUiObjects;
import de.duehl.vocabulary.japanese.ui.listcommander.selector.OwnListSelector;

/**
 * Diese Klasse lässt den Benutzer eine eigene Liste auswählen.
 *
 * @version 1.01     2025-11-29
 * @author Christian Dühl
 */

public class OwnListSelectionDialog extends ModalDialogBase {

    private static final Dimension DIALOG_DIMENSION = new Dimension(800, 300);

    /** Der anzuzeigende Text. */
    private final String text;

    /** Das Element zur Auswahl einer eigenen Liste. */
    private final OwnListSelector selector;

    /** Gibt an, ob der Dialog mit der Auswahl einer eigenen Liste beendet wurde. */
    private boolean applied;

    /**
     * Die ausgewählte Liste.
     *
     * Nur valide, wenn der Dialog mit der Auswahl einer eigenen Liste beendet wurde.
     */
    private OwnList ownList;

    /**
     * Konstruktor.
     *
     * @param title
     *            Der Titel des Dialoges.
     * @param text
     *            Der anzuzeigende Text.
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @param uiObjects
     *            Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers.
     * @param parentLocation
     *            Die Position des Rahmens der Oberfläche, vor der dieser Dialog angezeigt wird.
     */
    public OwnListSelectionDialog(String title, String text, FumikoDataStructures dataStructures,
            FumikoUiObjects uiObjects, Point parentLocation) {
        super(parentLocation, uiObjects.getProgramImage(), title, DIALOG_DIMENSION);

        this.text = text;

        Options options = dataStructures.getOptions();
        selector = new OwnListSelector(dataStructures, uiObjects,
                options.getHistoricalOwnLists(),
                ownList -> selectedOwnListChanged(ownList), () -> getLocation());
        selector.createSelector();

        applied = false;

        fillDialog();
    }

    private void selectedOwnListChanged(OwnList ownList) {
        this.ownList = ownList;
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        add(createTextPart(), BorderLayout.CENTER);
        add(createOwnListSelectorAndButtonPart(), BorderLayout.SOUTH);
    }

    private Component createTextPart() {
        String labelText = HtmlTool.htmlify(text);
        JLabel label = new JLabel(labelText);
        GuiTools.biggerFont(label, 10);
        label.setBorder(BorderFactory.createEmptyBorder(10, 25, 10, 25));
        return label;
    }

    private Component createOwnListSelectorAndButtonPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createOwnListSelectorPart(), BorderLayout.CENTER);
        panel.add(createButtonPart(), BorderLayout.SOUTH);

        return panel;
    }

    private Component createOwnListSelectorPart() {
        return selector.getPanel();
    }

    private Component createButtonPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(createSelectOwnListButton(), BorderLayout.CENTER);

        return panel;
    }

    private Component createSelectOwnListButton() {
        JButton button = new JButton("Angezeigte Liste auswählen");
        button.addActionListener(e -> selectOwnList());
        return button;
    }

    private void selectOwnList() {
        applied = true;
        closeDialog();
    }

    /** Gibt an, ob der Dialog mit der Auswahl einer eigenen Liste beendet wurde. */
    public boolean isApplied() {
        return applied;
    }

    /**
     * Getter für die ausgewählte Liste.
     *
     * Nur valide, wenn der Dialog mit der Auswahl einer eigenen Liste beendet wurde. Dies sollte
     * daher vorher mit der Methode isApplied() überprüft werden.
     */
    public OwnList getOwnList() {
        return ownList;
    }

}
