package de.duehl.vocabulary.japanese.ui.dialog.vocables.detail.addtolist;

import java.awt.Point;
import java.util.List;

import de.duehl.swing.ui.GuiTools;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.common.persistence.data.HistoricalOwnListPersistanceDataList;
import de.duehl.vocabulary.japanese.data.FumikoDataStructures;
import de.duehl.vocabulary.japanese.data.OwnList;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.logic.ownlists.OwnLists;
import de.duehl.vocabulary.japanese.ui.data.FumikoUiObjects;
import de.duehl.vocabulary.japanese.ui.dialog.ownlist.OwnListSelectionDialog;

/**
 * Diese Klasse fügt eine Vokabel zu einer eigenen Liste hinzu.
 *
 * @version 1.01     2025-11-20
 * @author Christian Dühl
 */

class VocableToOwnListAdder {

    /** Die Datenstrukturen des Vokabeltrainers. */
    private final FumikoDataStructures dataStructures;

    /** Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers. */
    private final FumikoUiObjects uiObjects;

    /**
     * Die Position des Rahmens der Oberfläche, die den Panel anzeigt, welcher diese Klasse
     * verwendet.
     */
    private final Point location;

    /**
     * Konstruktor.
     *
     * @param dataStructures
     *            Die Datenstrukturen des Vokabeltrainers.
     * @param uiObjects
     *            Die häufig verwendeten Funktionen der grafischen Oberfläche des Vokabeltrainers.
     * @param location
     *            Die Position des Rahmens der Oberfläche, die den Panel anzeigt, welcher diese
     *            Klasse verwendet.
     */
    public VocableToOwnListAdder(FumikoDataStructures dataStructures,
            FumikoUiObjects uiObjects, Point location) {
        this.dataStructures = dataStructures;
        this.uiObjects = uiObjects;
        this.location = location;
    }

    /**
     * Fügt die angezeigte Vokabel zu einer eigenen Liste hinzu.
     *
     * Dazu wählt der Benutzer eine eigene Liste aus. Wenn der Dialog mit dem Button "Angezeigte
     * Liste auswählen" beendet wurde und die gewählte Liste die Vokabel noch nicht enthält, wird
     * sie hinzugefügt.
     *
     * In diesem Erfolgsfall muss man die Listen speichern, da ja eine Vokabel hinzugefügt wurde.
     * Daher wird die Methode
     *     ownLists.storeOwnLists();
     * aufgerufen.
     * Man muss zwar nicht in der Gui die Listen neu aufbauen, denn an deren Anzahl und Namen wurde
     * nichts geändert, ebensowenig an der Anordnung der Listen etc., aber durch die geänderte
     * Füllung können sich die Farben und Prozentzahlen der Listen-Bars ändern. Daher könnte man
     * denken dass es reicht, die Methode
     *     ownLists.setCorrectForegroundColorOfVocabularyBarsInGui();
     * aufzurufen, durch welche die Farben und Prozente aktualisiert werden. Allerdings werden dann
     * bei einer vorher leeren Liste die Buttons nicht farbig und aktiv. Daher ruft man die
     * Methode
     *     ownLists.actualizeOwnListsPartInGui();
     * auf
     */
    public void addVocableToList(Vocable vocable) {
        String title = "Auswahl einer eigenen Liste";
        String text = "Bitte wählen Sie die eigene Liste aus, zu der die Vokabel "
                + vocable.getKanjiKanaRomajiWithJapaneseBraces() + " hinzugefügt werden soll.";
        OwnListSelectionDialog dialog = new OwnListSelectionDialog(title, text, dataStructures,
                uiObjects, location);
        dialog.setVisible(true);
        if (dialog.isApplied()) {
            OwnList ownList = dialog.getOwnList();
            Options options = dataStructures.getOptions();
            HistoricalOwnListPersistanceDataList historicalOwnLists =
                    options.getHistoricalOwnLists();
            historicalOwnLists.addOwnListData(ownList.toOwnListPersistanceData());

            List<Vocable> vocables = ownList.getVocables();
            if (vocables.contains(vocable)) {
                GuiTools.informUser("Die Vokabel ist in der Liste schon enthalten",
                        "Die Vokabel " + vocable.getKanjiKanaRomajiWithJapaneseBraces()
                            + " ist in der Liste '" + ownList.getName() + "' schon enthalten.");
            }
            else {
                vocables.add(vocable);
                GuiTools.informUser("Die Vokabel wurde zur Liste hinzugefügt",
                        "Die Vokabel " + vocable.getKanjiKanaRomajiWithJapaneseBraces()
                            + " wurde zur Liste '" + ownList.getName() + "' hinzugefügt.");
                OwnLists ownLists = dataStructures.getOwnLists();
                ownLists.storeOwnLists();
                ownLists.actualizeOwnListsPartInGui();
            }
        }
    }

}
